package com.xebialabs.xlrelease.security.sql

import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlrelease.security
import com.xebialabs.xlrelease.security.sql.db.Tables
import com.xebialabs.xlrelease.security.{ReleaseRolePermissionRepository, RolePermission}
import grizzled.slf4j.Logging
import slick.dbio.DBIOAction.seq
import slick.jdbc.JdbcProfile

import scala.annotation.varargs
import scala.jdk.CollectionConverters._

class SqlReleaseRolePermissionRepository(securityDatabase: Database) extends ReleaseRolePermissionRepository with Logging {

  import securityDatabase._

  val profile: JdbcProfile = config.databaseType.profile
  import profile.api._

  @varargs
  override def createGlobalRolePermissions(rolePermissions: RolePermission*): Unit = {
    val rows = rolePermissions.collect {
      case security.RolePermission(roleId, permissionNames) =>
        permissionNames.asScala.map(permissionName => Tables.RolePermission(roleId, permissionName))
    }.flatten
    val actions = seq(Tables.rolePermissions ++= rows)
    runAwait(actions.transactionally)
  }

  @varargs
  override def updateGlobalRolePermissions(rolePermissions: RolePermission*): Unit = {
    val rows = rolePermissions.collect {
      case security.RolePermission(roleId, permissionNames) =>
        permissionNames.asScala.map(permissionName => Tables.RolePermission(roleId, permissionName))
    }.flatten

    val actions: DBIOAction[_, NoStream, Effect.Write] = seq(
      ifNotEmpty(rolePermissions)(items => Seq(Tables.rolePermissions.filter(_.roleId in items.map(_.roleId)).delete)) ++
        ifNotEmpty(rows)(items => Seq(Tables.rolePermissions ++= items)): _*
    )

    runAwait(actions.transactionally)
  }

  private def ifNotEmpty[T, R](items: Iterable[T])(mapper: Iterable[T] => Seq[R]): Seq[R] = if (items.isEmpty) Seq.empty else mapper(items)

  override def getGlobalRolePermissions(): Seq[RolePermission] = {
    val query = for {
      (_, rp) <- Tables.roles.join(Tables.rolePermissions).on(_.id === _.roleId)
    } yield rp
    val rolePermissions = runAwait(query.result)
    rolePermissions.groupBy(_.roleId).collect {
      case (roleId, rps) => RolePermission(roleId, rps.collect(_.permissionName).toSet.asJava)
    }.toSeq
  }

  override def getRolePermissions(roleId: String): Set[String] = {
    val query = for {
      (_, rp) <- Tables.roles.filter(_.id === roleId).join(Tables.rolePermissions).on(_.id === _.roleId)
    } yield rp
    val rolePermissions = runAwait(query.result)
    rolePermissions.map(_.permissionName).toSet
  }
}
