package com.xebialabs.xlrelease.scheduler.logs

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASK_EXECUTIONS
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionEntry.taskExecutionRowMapper
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository.{FindOperation, UpdateTaskExecutionOperation}
import com.xebialabs.xlrelease.storage.domain.LogEntry
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

import java.time.Instant

@IsTransactional
class SqlTaskExecutionRepository(implicit val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends TaskExecutionRepository
    with PersistenceSupport
    with Logging
    with Utils {

  @Timed
  @IsReadOnly
  override def read(taskId: String, executionId: String): Option[TaskExecutionEntry] = {
    findOptional(_.queryForObject(STMT_FIND_TASK_EXECUTION, taskExecutionRowMapper, hash(taskId), executionId))
  }

  @Timed
  @IsReadOnly
  override def find(findOperation: FindOperation): Seq[TaskExecutionEntry] = {
    findOperation match {
      case TaskExecutionRepository.ByTaskId(taskId) =>
        findMany(sqlQuery(STMT_FIND_ALL_TASK_EXECUTIONS, params(TASK_EXECUTIONS.TASK_ID_HASH -> hash(taskId)), taskExecutionRowMapper))
    }
  }

  private def create(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_INSERT_ROW, row.asSqlParamMap())
    row
  }

  private def updateRow(row: TaskExecutionEntry): TaskExecutionEntry = {
    namedTemplate.update(STMT_UPDATE_ROW, row.asSqlParamMap())
    row
  }

  @Timed
  override def update(operation: UpdateTaskExecutionOperation): Option[TaskExecutionEntry] = {
    operation match {
      case TaskExecutionRepository.FinishExecution(taskId, executionId, endDate) =>
        val maybeRow = read(taskId, executionId)
        maybeRow match {
          case Some(row) => Some(updateRow(row.copy(endDate = endDate)))
          case None => None // Must be for a non container based task so do nothing
        }
      case TaskExecutionRepository.UpdateWithLogEntry(logEntry) =>
        val taskExecutionEntry = logEntryToTaskExecutionEntry(logEntry)
        val maybeRow = read(logEntry.taskId, logEntry.executionId)
        val row = maybeRow match {
          case Some(row) =>
            val updated = row.copy(
              // Always choose max values in case log entries are received out of order
              lastJob = Integer.max(taskExecutionEntry.lastJob, row.lastJob),
              lastChunk = Integer.max(taskExecutionEntry.lastChunk, row.lastChunk),
              lastModifiedDate = maxInstant(taskExecutionEntry.lastModifiedDate, row.lastModifiedDate)
            )
            updateRow(updated)
          case None =>
            create(taskExecutionEntry)
        }
        Some(row)
    }
  }

  private def maxInstant(a: Instant, b: Instant): Instant = {
    if (a.isAfter(b)) a else b
  }

  private def logEntryToTaskExecutionEntry(logEntry: LogEntry): TaskExecutionEntry = {
    TaskExecutionEntry(
      hash(logEntry.taskId),
      logEntry.executionId,
      logEntry.jobId.toInt,
      logEntry.chunk.toInt,
      Instant.parse(logEntry.lastEntryTimestamp),
      null
    )
  }

  @Timed
  override def delete(taskId: String, executionId: String): Unit = {
    val params = Map(
      TASK_EXECUTIONS.TASK_ID_HASH -> hash(taskId),
      TASK_EXECUTIONS.EXECUTION_ID -> executionId
    )
    namedTemplate.update(STMT_DELETE_ROW, params)
  }

  private val STMT_INSERT_ROW =
    s"""INSERT INTO ${TASK_EXECUTIONS.TABLE} (
       |  ${TASK_EXECUTIONS.TASK_ID_HASH},
       |  ${TASK_EXECUTIONS.EXECUTION_ID},
       |  ${TASK_EXECUTIONS.LAST_JOB},
       |  ${TASK_EXECUTIONS.LAST_CHUNK},
       |  ${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  ${TASK_EXECUTIONS.END_DATE}
       | ) VALUES (
       |  :${TASK_EXECUTIONS.TASK_ID_HASH},
       |  :${TASK_EXECUTIONS.EXECUTION_ID},
       |  :${TASK_EXECUTIONS.LAST_JOB},
       |  :${TASK_EXECUTIONS.LAST_CHUNK},
       |  :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |  :${TASK_EXECUTIONS.END_DATE}
       | )
       |""".stripMargin

  private val STMT_UPDATE_ROW =
    s"""UPDATE ${TASK_EXECUTIONS.TABLE}
       |  SET
       |    ${TASK_EXECUTIONS.LAST_JOB} = :${TASK_EXECUTIONS.LAST_JOB},
       |    ${TASK_EXECUTIONS.LAST_CHUNK} = :${TASK_EXECUTIONS.LAST_CHUNK},
       |    ${TASK_EXECUTIONS.LAST_MODIFIED_DATE} = :${TASK_EXECUTIONS.LAST_MODIFIED_DATE},
       |    ${TASK_EXECUTIONS.END_DATE} = :${TASK_EXECUTIONS.END_DATE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_DELETE_ROW =
    s"""DELETE FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE
       |    ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  AND
       |    ${TASK_EXECUTIONS.EXECUTION_ID} = :${TASK_EXECUTIONS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_FIND_TASK_EXECUTION =
    s"""SELECT * FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE ${TASK_EXECUTIONS.TASK_ID_HASH} = ?
       |  AND ${TASK_EXECUTIONS.EXECUTION_ID} = ?
       |""".stripMargin

  private val STMT_FIND_ALL_TASK_EXECUTIONS =
    s"""SELECT * FROM ${TASK_EXECUTIONS.TABLE}
       |  WHERE ${TASK_EXECUTIONS.TASK_ID_HASH} = :${TASK_EXECUTIONS.TASK_ID_HASH}
       |  ORDER BY ${TASK_EXECUTIONS.LAST_MODIFIED_DATE} ASC
       |""".stripMargin
}
