package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.UserProfileRepository
import com.xebialabs.xlrelease.security.sql.SecurityUserCacheConfigurationCondition
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable, Caching}
import org.springframework.context.annotation.Conditional
import org.springframework.stereotype.Component

import java.lang
import java.util.{Date, List => JList}

@CacheConfig(cacheManager = SECURITY_USER_CACHE_MANAGER)
class CachingSqlUserProfileRepositoryWrapper(repository: UserProfileRepository, evicter: UserProfileCacheEvicter) extends UserProfileRepository with Logging {

  override def create(userProfile: UserProfile): Unit = {
    repository.create(userProfile)
    evicter.onUserProfileCacheEntryCreated(UserProfileCacheEntryCreated(userProfile.getCanonicalId))
  }

  override def update(userProfile: JList[UserProfile]): Unit = {
    repository.update(userProfile)
    evicter.onMultipleUserProfileCacheEntriesUpdated(MultipleUserProfileCacheEntriesUpdated())
  }

  override def updateLastActive(canonicalId: String, lastActive: Date): Boolean = repository.updateLastActive(canonicalId, lastActive)

  override def updateLastActiveBatch(entries: Map[String, Date]): Int = repository.updateLastActiveBatch(entries)

  override def delete(userProfileId: String): Unit = {
    repository.delete(userProfileId)
    evicter.onUserProfileCacheEntryDeleted(UserProfileCacheEntryDeleted(userProfileId))
  }

  @Cacheable(cacheNames = Array(SECURITY_USER_PROFILE), key = "#canonicalId", unless = "#result.isEmpty")
  override def findById(canonicalId: String): Option[UserProfile] = repository.findById(canonicalId)

  @Cacheable(cacheNames = Array(SECURITY_ALL_USER_PROFILES))
  override def findAll(fullProfile: Boolean): JList[UserProfile] = repository.findAll(fullProfile)

  override def exists(canonicalId: String): Boolean = repository.exists(canonicalId)

  override def customSearch(email: String, fullName: String, loginAllowed: lang.Boolean, lastActiveAfter: Date, lastActiveBefore: Date,
                            page: Option[Long], resultsPerPage: Option[Long]): JList[UserProfile] =
    repository.customSearch(email, fullName, loginAllowed, lastActiveAfter, lastActiveBefore, page, resultsPerPage)

  override def countUserWithLoginAllowed(): Int = repository.countUserWithLoginAllowed()

}

case class UserProfileCacheEntryCreated(canonicalId: String, publish: Boolean = true) extends DistributedXLReleaseEvent
case class MultipleUserProfileCacheEntriesUpdated(publish: Boolean = true) extends DistributedXLReleaseEvent
case class UserProfileCacheEntryDeleted(canonicalId: String, publish: Boolean = true) extends DistributedXLReleaseEvent

@Component
@EventListener
@Conditional(value = Array(classOf[SecurityUserCacheConfigurationCondition]))
@CacheConfig(cacheManager = SECURITY_USER_CACHE_MANAGER)
class UserProfileCacheEvicter(broadcastService: BroadcastService) {

  @AsyncSubscribe
  @CacheEvict(cacheNames = Array(SECURITY_ALL_USER_PROFILES), allEntries = true)
  def onUserProfileCacheEntryCreated(event: UserProfileCacheEntryCreated): Unit = {
    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }

  @AsyncSubscribe
  @CacheEvict(cacheNames = Array(SECURITY_USER_PROFILE, SECURITY_ALL_USER_PROFILES), allEntries = true)
  def onMultipleUserProfileCacheEntriesUpdated(event: MultipleUserProfileCacheEntriesUpdated): Unit = {
    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }

  @AsyncSubscribe
  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_USER_PROFILE), key = "#event.canonicalId"),
    new CacheEvict(cacheNames = Array(SECURITY_ALL_USER_PROFILES), allEntries = true)
  ))
  def onUserProfileCacheEntryDeleted(event: UserProfileCacheEntryDeleted): Unit = {
    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }
}
