package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.DatabaseInfo
import com.xebialabs.xlrelease.db.sql.DatabaseInfo._
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import java.sql.ResultSet
import java.util.Date

private[persistence] object NotificationQueries {

  trait DialectFunctions {
    def addDuration(duration: String, startDate: String): String

    def diffDates(startDate: String, endDate: String): String

    def castToInt(expr: String): String
  }

  /**
   * @see <a href="https://db.apache.org/derby/docs/10.1/ref">https://db.apache.org/derby/docs/10.1/ref/</a>
   */
  object DerbyDialectFunctions extends DialectFunctions {
    def addDuration(duration: String, startDate: String): String = {
      s"{fn TIMESTAMPADD(SQL_TSI_SECOND, $duration, $startDate)}"
    }

    def diffDates(startDate: String, endDate: String): String = {
      s"{fn TIMESTAMPDIFF(SQL_TSI_SECOND, $startDate, $endDate)}"
    }

    def castToInt(expr: String): String = {
      s"CAST ($expr AS INTEGER)"
    }
  }

  /**
   * @see <a href="https://docs.microsoft.com/en-us/sql/t-sql/data-types/date-and-time-types?view=sql-server-ver15">https://docs.microsoft.com/en-us/sql/t-sql/data-types/date-and-time-types?view=sql-server-ver15</a>
   */
  object MsSqlDialectFunctions extends DialectFunctions {
    def addDuration(duration: String, startDate: String): String = {
      s"DATEADD(second, $duration, $startDate)"
    }

    def diffDates(startDate: String, endDate: String): String = {
      s"DATEDIFF(second, $startDate, $endDate)"
    }

    override def castToInt(expr: String): String = {
      s"CAST ($expr AS INT)"
    }
  }

  /**
   * @see <a href="https://dev.mysql.com/doc/refman/8.0/en/">https://dev.mysql.com/doc/refman/8.0/en/</a>
   */
  object MySqlDialectFunctions extends DialectFunctions {
    override def addDuration(duration: String, startDate: String): String = {
      s"(($startDate) + INTERVAL $duration SECOND)"
    }

    override def diffDates(startDate: String, endDate: String): String = {
      s"(TIMESTAMPDIFF(second, $startDate, $endDate))"
    }

    override def castToInt(expr: String): String = {
      s"(CAST($expr AS SIGNED))"
    }
  }

  /**
   * @see <a href="https://www.postgresql.org/docs/9.5/index.html">https://www.postgresql.org/docs/9.5/index.html</a>
   */
  object PostgreSQLDialectFunctions extends DialectFunctions {
    override def addDuration(duration: String, startDate: String): String = {
      s"$startDate + $duration * INTERVAL '1' SECOND"
    }

    override def diffDates(startDate: String, endDate: String): String = {
      s"EXTRACT(epoch FROM $endDate - $startDate)"
    }

    override def castToInt(expr: String): String = {
      s"CAST($expr AS INT)"
    }
  }

  /**
   * @see <a href="https://www.ibm.com/support/knowledgecenter/SSEPGG_9.7.0/com.ibm.db2.luw.kc.doc/welcome.html">https://www.ibm.com/support/knowledgecenter/SSEPGG_9.7.0/com.ibm.db2.luw.kc.doc/welcome.html</a>
   */
  object Db2DialectFunctions extends DialectFunctions {
    override def addDuration(duration: String, startDate: String): String = {
      s"$startDate + $duration SECONDS"
    }

    override def diffDates(startDate: String, endDate: String): String = {
      // https://www.ibm.com/support/knowledgecenter/SSEPGG_9.7.0/com.ibm.db2.luw.sql.ref.doc/doc/r0000861.html
      s"TIMESTAMPDIFF(2, CHAR($endDate - $startDate))"
    }

    override def castToInt(expr: String): String = {
      // https://www.ibm.com/support/knowledgecenter/SSEPGG_9.7.0/com.ibm.db2.luw.sql.ref.doc/doc/r0023459.html
      s"CAST($expr AS INT)"
    }
  }

  object OracleDialectFunctions extends DialectFunctions {
    override def addDuration(duration: String, startDate: String): String = {
      s"$startDate + $duration * INTERVAL '1' SECOND"
    }

    override def diffDates(startDate: String, endDate: String): String = {
      s"""EXTRACT(second FROM $endDate - $startDate)
         |+ EXTRACT(minute FROM $endDate - $startDate) * 60
         |+ EXTRACT(hour FROM $endDate - $startDate) * 60 * 60
         |+ EXTRACT(day FROM $endDate - $startDate) * 60 * 60 * 24
         |""".stripMargin
    }

    override def castToInt(expr: String): String = {
      s"TO_NUMBER ($expr)"
    }
  }

  object H2DialectFunctions extends DialectFunctions {
    override def addDuration(duration: String, startDate: String): String = {
      s"DATEADD(second, $duration, $startDate)"
    }

    override def diffDates(startDate: String, endDate: String): String = {
      // http://www.h2database.com/html/functions.html#datediff
      s"TIMESTAMPDIFF(second, $startDate, $endDate)"
    }

    override def castToInt(expr: String): String = {
      s"CAST ($expr AS INT)"
    }
  }

  trait BaseNotificationQuery[T] extends Logging {

    import scala.jdk.CollectionConverters._

    val dialectFunctions: DialectFunctions = dbInfo match {
      case Derby(_) => DerbyDialectFunctions
      case MsSqlServer(_) => MsSqlDialectFunctions
      case Oracle(_) => OracleDialectFunctions
      case MySql(_) => MySqlDialectFunctions
      case Db2(_) => Db2DialectFunctions
      case PostgreSql(_) => PostgreSQLDialectFunctions
      case H2(_) => H2DialectFunctions
      case TestDatabaseInfo(_) => H2DialectFunctions
      case Unknown(metadata) => throw new IllegalStateException(s"Unknown database ${metadata.getDatabaseProductName}:${metadata.getDatabaseProductVersion}")
    }

    def dbInfo: DatabaseInfo

    def namedTemplate: NamedParameterJdbcTemplate

    protected def mapper: RowMapper[T]

    protected def stmt: String

    private val queryStmt = stmt

    def execute: Seq[T] = {
      logger.trace(s"Executing query: $queryStmt")
      namedTemplate.query[T](queryStmt, Map(("CURRENT_TIMESTAMP", new Date())).asJava, mapper).asScala.toSeq
    }
  }

  trait TaskNotificationQuery {
    def taskMapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
      val shortTaskId = rs.getString(TASKS.TASK_ID)
      val fullFolderId = (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute
      s"$fullFolderId/$shortTaskId"
    }

    def mapper: RowMapper[String] = taskMapper
  }

  class DueSoonTasksQuery(val dbInfo: DatabaseInfo, val namedTemplate: NamedParameterJdbcTemplate)
    extends BaseNotificationQuery[String] with TaskNotificationQuery {

    override protected def stmt: String = {
      val DUE_SOON_CANDIDATE_STATUSES = TaskStatus.ACTIVE_STATUSES.map(_.value())
      val DUE_SOON_RELEASE_CANDIDATE_STATUSES = Seq(ReleaseStatus.PLANNED.value()) ++ ReleaseStatus.ACTIVE_STATUSES.map(_.value())

      import dialectFunctions._
      def cutoffDate(endDate: String): String = addDuration(castToInt(s"${Task.DUE_SOON_THRESHOLD} * (${diffDates(TASKS.START_DATE, endDate)})"), TASKS.START_DATE)

      val calculatedEndDate: String = addDuration(TASKS.PLANNED_DURATION, TASKS.START_DATE)

      s"""SELECT t.${TASKS.TASK_ID}, f.${FOLDERS.FOLDER_ID}, f.${FOLDERS.FOLDER_PATH}
         | FROM
         |       ${RELEASES.TABLE} r
         |       JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
         |       JOIN (SELECT ${TASKS.TASK_ID},
         |             CASE
         |                 WHEN ${TASKS.END_DATE} IS NOT NULL THEN CASE WHEN :CURRENT_TIMESTAMP BETWEEN ${cutoffDate(TASKS.END_DATE)} AND ${TASKS.END_DATE} THEN 1 ELSE 0 END
         |                 WHEN ${TASKS.PLANNED_DURATION} > 0 AND :CURRENT_TIMESTAMP BETWEEN ${cutoffDate(calculatedEndDate)} AND $calculatedEndDate THEN 1
         |                 ELSE 0
         |             END IS_DUE_SOON,
         |             ${TASKS.IS_DUE_SOON_NOTIFIED},
         |             ${TASKS.RELEASE_UID}
         |      FROM ${TASKS.TABLE}
         |      WHERE ${TASKS.STATUS} IN (${DUE_SOON_CANDIDATE_STATUSES.mkString("'", "','", "'")})
         |     ) t ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
         | WHERE
         |  r.${RELEASES.STATUS} IN (${DUE_SOON_RELEASE_CANDIDATE_STATUSES.mkString("'", "','", "'")})
         |  AND t.IS_DUE_SOON = 1
         |  AND t.IS_DUE_SOON_NOTIFIED = 0""".stripMargin
    }
  }

  class OverdueTasksQuery(val dbInfo: DatabaseInfo, val namedTemplate: NamedParameterJdbcTemplate)
    extends BaseNotificationQuery[String] with TaskNotificationQuery {

    override protected def stmt: String = {
      import dialectFunctions._
      val OVERDUE_CANDIDATE_STATUSES = Seq(TaskStatus.PLANNED.value()) ++ TaskStatus.ACTIVE_STATUSES.map(_.value())
      val OVERDUE_RELEASE_CANDIDATE_STATUSES = Seq(ReleaseStatus.PLANNED.value()) ++ ReleaseStatus.ACTIVE_STATUSES.map(_.value())
      s"""SELECT t.${TASKS.TASK_ID}, f.${FOLDERS.FOLDER_ID}, f.${FOLDERS.FOLDER_PATH}
         | FROM
         |       ${RELEASES.TABLE} r
         |       JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
         |       JOIN (SELECT ${TASKS.TASK_ID},
         |             CASE
         |                 WHEN ${TASKS.END_DATE} IS NOT NULL THEN CASE WHEN ${TASKS.END_DATE} <= :CURRENT_TIMESTAMP THEN 1 ELSE 0 END
         |                 WHEN ${TASKS.PLANNED_DURATION} > 0 AND ((${addDuration(TASKS.PLANNED_DURATION, TASKS.START_DATE)}) <= :CURRENT_TIMESTAMP) THEN 1
         |                 ELSE 0
         |             END IS_OVERDUE,
         |             ${TASKS.IS_OVERDUE_NOTIFIED},
         |             ${TASKS.RELEASE_UID}
         |      FROM ${TASKS.TABLE}
         |      WHERE ${TASKS.STATUS} IN (${OVERDUE_CANDIDATE_STATUSES.mkString("'", "','", "'")})
         |     ) t ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
         | WHERE
         |  r.${RELEASES.STATUS} IN (${OVERDUE_RELEASE_CANDIDATE_STATUSES.mkString("'", "','", "'")})
         |  AND t.IS_OVERDUE = 1
         |  AND t.IS_OVERDUE_NOTIFIED = 0""".stripMargin
    }

  }

  class OverdueReleasesQuery(val dbInfo: DatabaseInfo, val namedTemplate: NamedParameterJdbcTemplate) extends BaseNotificationQuery[String] {

    override protected def stmt: String = {
      import dialectFunctions._
      val OVERDUE_CANDIDATE_STATUSES = Seq(ReleaseStatus.PLANNED.value()) ++ ReleaseStatus.ACTIVE_STATUSES.map(_.value())
      s"""SELECT t.${RELEASES.RELEASE_ID}, f.${FOLDERS.FOLDER_ID}, f.${FOLDERS.FOLDER_PATH}
         |FROM (SELECT ${RELEASES.RELEASE_ID},
         |             CASE
         |                 WHEN ${RELEASES.END_DATE} IS NOT NULL THEN CASE WHEN ${RELEASES.END_DATE} <= :CURRENT_TIMESTAMP THEN 1 ELSE 0 END
         |                 WHEN ${RELEASES.PLANNED_DURATION} > 0 AND ((${addDuration(RELEASES.PLANNED_DURATION, RELEASES.START_DATE)}) <= :CURRENT_TIMESTAMP) THEN 1
         |                 ELSE 0
         |             END IS_OVERDUE,
         |             ${RELEASES.IS_OVERDUE_NOTIFIED},
         |             ${RELEASES.FOLDER_CI_UID}
         |      FROM ${RELEASES.TABLE}
         |      WHERE ${RELEASES.STATUS} IN (${OVERDUE_CANDIDATE_STATUSES.mkString("'", "','", "'")})
         |     ) t
         |         JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = t.${RELEASES.FOLDER_CI_UID}
         |WHERE t.IS_OVERDUE = 1
         |  AND t.IS_OVERDUE_NOTIFIED = 0""".stripMargin
    }

    override def mapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
      val shortReleaseId = rs.getString(RELEASES.RELEASE_ID)
      val fullFolderId = (FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)).absolute
      s"$fullFolderId/$shortReleaseId"
    }
  }

}
