package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Category, Release}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CATEGORIES, RELEASE_CATEGORY_REFS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder.STMT_CATEGORY_SELECT
import com.xebialabs.xlrelease.repository.sql.{SqlRepository, persistence}
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import scala.jdk.CollectionConverters._

@IsTransactional
class CategoryPersistence(implicit val jdbcTemplate: JdbcTemplate,
                          implicit val dialect: Dialect)
  extends SqlRepository
    with CategoryMapper
    with PersistenceSupport
    with Logging
    with Utils {

  private val STMT_INSERT_CATEGORY =
    s"""INSERT INTO ${CATEGORIES.TABLE} (
       |  ${CATEGORIES.CATEGORY},
       |  ${CATEGORIES.ACTIVE}
       | ) VALUES (
       |  :${CATEGORIES.CATEGORY},
       |  :${CATEGORIES.ACTIVE}
       | )
       |""".stripMargin

  private def checkTitleLength(title: String): Unit = {
    if (title.length > persistence.INDEXED_VARCHAR_LENGTH) {
      throw new IllegalArgumentException(s"The category title must be ${persistence.INDEXED_VARCHAR_LENGTH} characters or less")
    }
  }

  def create(category: Category): Category = {
    logger.trace(s"creating category $category")
    checkTitleLength(category.getTitle)
    val params = Map(
      CATEGORIES.CATEGORY -> category.getTitle,
      CATEGORIES.ACTIVE -> category.getActive.asInteger
    )

    val ciUid = sqlInsert(pkName(CATEGORIES.CI_UID), STMT_INSERT_CATEGORY, params)

    category.setCiUid(ciUid)
    category
  }

  private val STMT_UPDATE_CATEGORY =
    s"""UPDATE ${CATEGORIES.TABLE}
       |  SET
       |    ${CATEGORIES.CATEGORY} = :${CATEGORIES.CATEGORY},
       |    ${CATEGORIES.ACTIVE} = :${CATEGORIES.ACTIVE}
       |  WHERE ${CATEGORIES.CI_UID} = :${CATEGORIES.CI_UID}
       |""".stripMargin

  def update(category: Category): Category = {
    logger.trace(s"updating category $category")
    checkTitleLength(category.getTitle)
    val params = Map(
      CATEGORIES.CATEGORY -> category.getTitle,
      CATEGORIES.ACTIVE -> category.getActive.asInteger,
      CATEGORIES.CI_UID -> category.getCiUid
    )

    sqlUpdate(STMT_UPDATE_CATEGORY, params, _ => ())
    category
  }

  private val STMT_DELETE_CATEGORY = {
    s"DELETE FROM ${CATEGORIES.TABLE} WHERE ${CATEGORIES.CI_UID} = :${CATEGORIES.CI_UID}"
  }

  def delete(id: Int): Unit = {
    logger.trace(s"deleting category with id[$id]")
    sqlExec(STMT_DELETE_CATEGORY, params(CATEGORIES.CI_UID -> id), _.execute())
  }

  private val STMT_FIND_CATEGORY_BY_ID =
    s"""
       |$STMT_CATEGORY_SELECT
       | WHERE ${CATEGORIES.CI_UID} = ?
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(System.lineSeparator)

  def findByCiUid(id: Int): Option[Category] = {
    findOptional(_.queryForObject(STMT_FIND_CATEGORY_BY_ID, categoryMapper, id))
  }

  private val STMT_FIND_CATEGORY_BY_TITLE =
    s"""
       |$STMT_CATEGORY_SELECT
       | WHERE LOWER(${CATEGORIES.CATEGORY}) = ?
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(System.lineSeparator)


  def findByTitle(title: String): Option[Category] = {
    val searchableTitle = Category.sanitizeTitle(title).toLowerCase()
    findOptional(_.queryForObject(STMT_FIND_CATEGORY_BY_TITLE, categoryMapper, searchableTitle))
  }

  private val STMT_FIND_CATEGORIES_BY_TITLE =
    s"""
       |$STMT_CATEGORY_SELECT
       | WHERE LOWER(${CATEGORIES.CATEGORY}) IN (:titles)
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(System.lineSeparator)

  def findByTitles(titles: Seq[String]): Seq[Category] = {
    findMany(
      sqlQuery(STMT_FIND_CATEGORIES_BY_TITLE,
        params("titles" -> titles.map(Category.sanitizeTitle(_).toLowerCase()).asJava),
        categoryMapper)
    )
  }

  def findBy(categoryFilters: CategoryFilters, pageable: Pageable): Page[Category] = {
    CategoryQueryBuilder(dialect, namedTemplate)
      .from(categoryFilters)
      .withPageable(pageable)
      .build()
      .execute()
  }

  def decorate(release: Release): Release = {
    if (release.isTemplate) {
      val categories = findTitlesByReleaseCiUid(release.getCiUid).asJava
      release.setCategories(categories)
    }

    release
  }

  private val STMT_FIND_CATEGORIES_BY_RELEASE_UID: String =
    s"""
       |SELECT
       |  ${CATEGORIES.CATEGORY}
       | FROM ${CATEGORIES.TABLE} c
       | JOIN ${RELEASE_CATEGORY_REFS.TABLE} r ON c.${CATEGORIES.CI_UID} = r.${RELEASE_CATEGORY_REFS.targetColumn}
       | WHERE r.${RELEASE_CATEGORY_REFS.sourceColumn} = :${RELEASE_CATEGORY_REFS.sourceColumn}
       |""".stripMargin

  def findTitlesByReleaseCiUid(releaseUid: Int): Set[String] = {
    findMany(
      sqlQuery(STMT_FIND_CATEGORIES_BY_RELEASE_UID,
        Utils.params(RELEASE_CATEGORY_REFS.sourceColumn -> releaseUid),
        rs => rs.getString(CATEGORIES.CATEGORY))
    ).toSet
  }

  private val STMT_FIND_ACTIVE_CATEGORIES_BY_RELEASE_UIDS: String =
    s"""
       |SELECT
       |  r.${RELEASE_CATEGORY_REFS.sourceColumn},
       |  c.${CATEGORIES.CATEGORY}
       | FROM ${CATEGORIES.TABLE} c
       | JOIN ${RELEASE_CATEGORY_REFS.TABLE} r ON c.${CATEGORIES.CI_UID} = r.${RELEASE_CATEGORY_REFS.targetColumn}
       | WHERE
       |  r.${RELEASE_CATEGORY_REFS.sourceColumn} IN (:${RELEASE_CATEGORY_REFS.sourceColumn})
       | AND c.${CATEGORIES.ACTIVE} = 1
       |""".stripMargin

  def findActiveCategoryByReleaseCiUids(releaseCiUids: Set[Integer]): Map[Int, Set[String]] = {
    val activeCategories = sqlQuery(
      STMT_FIND_ACTIVE_CATEGORIES_BY_RELEASE_UIDS,
      params(RELEASE_CATEGORY_REFS.sourceColumn -> releaseCiUids.asJava),
      (rs, _) => (rs.getInt(RELEASE_CATEGORY_REFS.sourceColumn), rs.getString(CATEGORIES.CATEGORY))
    ).toSet

    activeCategories.groupMap(_._1)(_._2)
  }
}

trait CategoryMapper {
  val categoryMapper: RowMapper[Category] = (rs, _) => {
    val category: Category = new Category()
    category.setCiUid(rs.getInt(CATEGORIES.CI_UID))
    category.setTitle(rs.getString(CATEGORIES.CATEGORY))
    category.setActive(rs.getInt(CATEGORIES.ACTIVE).asBoolean)
    category
  }
}
