package com.xebialabs.xlrelease.spring.config

import com.xebialabs.xlplatform.repository.sql.{Database, DatabaseConfig, DatabaseType}
import com.xebialabs.xlplatform.upgrade.sql.{SqlInitializationStrategy, SqlRepositoryVersionService, SqlUpgradeStrategy}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.XLReleaseDbInitializer
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.db.sql.{DatabaseInfo, SqlDialectDetector}
import com.xebialabs.xlrelease.features.DatabaseProxyFeature
import com.xebialabs.xlrelease.metrics.XlrMetricRegistry
import com.xebialabs.xlrelease.repository.sql.JdbcTemplateHolder
import com.xebialabs.xlrelease.service.SqlCiIdService
import com.xebialabs.xlrelease.spring.config.SqlConfiguration.{XLR_REPORTING_DATA_SOURCE, XLR_REPOSITORY_DATA_SOURCE, getDatabaseType}
import com.zaxxer.hikari.{HikariConfig, HikariDataSource}
import grizzled.slf4j.Logging
import org.springframework.context.annotation._
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.datasource.{DataSourceTransactionManager, TransactionAwareDataSourceProxy}
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.TransactionManagementConfigurer
import org.springframework.transaction.support.TransactionTemplate

import java.util.concurrent.{Executors, ThreadFactory}
import javax.sql.DataSource

//noinspection ScalaStyle
@Configuration
class SqlConfiguration(xlrConfig: XlrConfig, databaseProxyFeature: DatabaseProxyFeature) extends TransactionManagementConfigurer with Logging {

  @Bean
  def jdbcTemplateHolder(): JdbcTemplateHolder.type = {
    JdbcTemplateHolder.init(xlrRepositoryJdbcTemplate())
    JdbcTemplateHolder
  }

  @Bean
  def ciIdService() = new SqlCiIdService()

  @Bean
  def upgradeStrategy = new SqlUpgradeStrategy()

  @Bean
  def initializationStrategy = new SqlInitializationStrategy()

  @Bean
  def repositoryVersionService() = new SqlRepositoryVersionService(xlrRepositoryDatabase())

  @Bean
  def xlrMigrationsDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlplatform/upgrade/sql/db/xl-migrations-schema.yaml", xlrRepositoryDataSourceProxy())
  }

  @Bean
  def xlrRepositoryDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlrelease/db/xl-release-schema-initializer.yaml", xlrRepositoryDataSourceProxy())
  }

  @Bean
  def rawRepositoryDataSource(): HikariDataSource = {
    new HikariDataSource(xlrRepositoryDataSourceConfig(xlrConfig))
  }

  @Bean(destroyMethod = "close", name = Array(XLR_REPOSITORY_DATA_SOURCE))
  def xlrRepositoryDataSource(): CloseableDataSource = {
    new XlrDelegatingDataSource(rawRepositoryDataSource(), databaseProxyFeature, xlrConfig)
  }

  @Bean
  def xlrRepositoryDataSourceProxy(): DataSource = {
    new TransactionAwareDataSourceProxy(xlrRepositoryDataSource())
  }

  @Bean
  def xlrRepositoryDatabase(): Database = {
    val databaseConfig = DatabaseConfig(
      getDatabaseType(xlrRepositorySqlDialect()),
      "xlr-ds",
      minThreads = xlrConfig.database.maxPoolSize,
      maxThreads = xlrConfig.database.maxPoolSize,
      maxConnections = xlrConfig.database.maxPoolSize,
      queryTimeout = xlrConfig.timeouts.releaseActionResponse
    )

    Database(xlrRepositoryDataSourceProxy(), databaseConfig)
  }

  override def annotationDrivenTransactionManager(): PlatformTransactionManager = xlrRepositoryTransactionManager()

  @Bean
  def xlrRepositoryTransactionManager(): PlatformTransactionManager = {
    val txManager = new DataSourceTransactionManager()
    txManager.setDataSource(xlrRepositoryDataSource())
    txManager
  }

  @Bean
  def xlrRepositoryTransactionTemplate(): TransactionTemplate = {
    new TransactionTemplate(xlrRepositoryTransactionManager())
  }

  @Bean
  def xlrRepositoryJdbcTemplate(): JdbcTemplate = {
    new JdbcTemplate(xlrRepositoryDataSource())
  }

  @Bean
  def xlrDatabaseInformation: DatabaseInfo = {
    DatabaseInfo(rawRepositoryDataSource())
  }

  @Bean
  def xlrDbInfo(): DatabaseInfo = {
    val dbInfo = DatabaseInfo(rawRepositoryDataSource())
    logger.info(s"Repository database is ${dbInfo.dbName}:${dbInfo.dbVersion}")
    dbInfo
  }

  @Bean
  def xlrRepositoryDialectDetector(): SqlDialectDetector = {
    new SqlDialectDetector()
  }

  @Bean
  def xlrRepositorySqlDialect(): Dialect = {
    xlrRepositoryDialectDetector().detectSqlDialect(xlrDbInfo())
  }

  @Bean
  def xlrPrivilegedThreadFactory(): ThreadFactory =
    Executors.privilegedThreadFactory()

  // Report database
  @Bean
  def rawReportingDataSource(): HikariDataSource = {
    new HikariDataSource(reportingDataSourceConfig(xlrConfig))
  }

  @Bean(destroyMethod = "close", name = Array(XLR_REPORTING_DATA_SOURCE))
  def reportingDataSource(): CloseableDataSource = {
    new XlrDelegatingDataSource(rawReportingDataSource(), databaseProxyFeature, xlrConfig)
  }

  @Bean
  def reportingDataSourceProxy(): DataSource = {
    new TransactionAwareDataSourceProxy(reportingDataSource())
  }

  @Bean
  def reportingDbInitializer(): XLReleaseDbInitializer = {
    new XLReleaseDbInitializer("com/xebialabs/xlrelease/db/archivingTablesInitialize.yaml", reportingDataSourceProxy())
  }

  @Bean
  def reportingTransactionManager(): PlatformTransactionManager = {
    val txManager = new DataSourceTransactionManager()
    txManager.setDataSource(reportingDataSource())
    txManager
  }

  @Bean
  def reportTransactionTemplate(): TransactionTemplate = {
    new TransactionTemplate(reportingTransactionManager())
  }

  @Bean
  def reportingJdbcTemplate(): JdbcTemplate = {
    new JdbcTemplate(reportingDataSource())
  }

  @Bean
  def reportingDialectDetector(): SqlDialectDetector = {
    new SqlDialectDetector()
  }

  @Bean
  def reportingSqlDialect(): Dialect = {
    val jdbcTemplate = new JdbcTemplate(rawReportingDataSource())
    reportingDialectDetector().detectSqlDialect(jdbcTemplate)
  }

  private def xlrRepositoryDataSourceConfig(xlrConfig: XlrConfig): HikariConfig = {
    val hc = new HikariConfig()
    hc.setJdbcUrl(xlrConfig.database.dbUrl)
    hc.setDriverClassName(xlrConfig.database.dbDriverClassname)
    hc.setUsername(xlrConfig.database.dbUsername)
    hc.setPassword(xlrConfig.database.dbPassword)
    hc.setPoolName(xlrConfig.database.poolName)
    hc.setMaximumPoolSize(xlrConfig.database.maxPoolSize)
    hc.setConnectionTimeout(xlrConfig.database.connectionTimeout)
    hc.setMaxLifetime(xlrConfig.database.maxLifeTime)
    hc.setIdleTimeout(xlrConfig.database.idleTimeout)
    hc.setMinimumIdle(xlrConfig.database.minimumIdle)
    hc.setLeakDetectionThreshold(xlrConfig.database.leakConnectionThreshold)
    hc.setMetricRegistry(XlrMetricRegistry.metricRegistry)
    hc.setTransactionIsolation("TRANSACTION_READ_COMMITTED")
    hc.setThreadFactory(xlrPrivilegedThreadFactory())
    hc.setAllowPoolSuspension(true)
    hc
  }

  private def reportingDataSourceConfig(xlrConfig: XlrConfig): HikariConfig = {
    val hc = new HikariConfig()
    hc.setJdbcUrl(xlrConfig.reporting.dbUrl)
    hc.setDriverClassName(xlrConfig.reporting.dbDriverClassname)
    hc.setUsername(xlrConfig.reporting.dbUsername)
    hc.setPassword(xlrConfig.reporting.dbPassword)
    hc.setMaximumPoolSize(xlrConfig.reporting.maxPoolSize)
    hc.setPoolName(xlrConfig.reporting.poolName)
    hc.setConnectionTimeout(xlrConfig.reporting.connectionTimeout)
    hc.setMaxLifetime(xlrConfig.reporting.maxLifeTime)
    hc.setIdleTimeout(xlrConfig.reporting.idleTimeout)
    hc.setMinimumIdle(xlrConfig.reporting.minimumIdle)
    hc.setLeakDetectionThreshold(xlrConfig.reporting.leakConnectionThreshold)
    hc.setMetricRegistry(XlrMetricRegistry.metricRegistry)
    hc.setThreadFactory(xlrPrivilegedThreadFactory())
    hc.setAllowPoolSuspension(true)
    hc
  }
}

object SqlConfiguration {
  final val XLR_REPOSITORY_DATA_SOURCE = "xlrRepositoryDataSource"
  final val XLR_REPORTING_DATA_SOURCE = "reportingDataSource"

  def getDatabaseType(dialect: Dialect): DatabaseType with Serializable = dialect match {
    case DerbyDialect(_) => DatabaseType.Derby
    case OracleDialect(_) => DatabaseType.Oracle
    case Db2Dialect(_) => DatabaseType.DB2
    case MSSQLDialect(_) => DatabaseType.SQLServer
    case CommonDialect(dbName) if dbName.contains("h2") => DatabaseType.H2
    case CommonDialect(dbName) if dbName.contains("mysql") => DatabaseType.MySQL
    case CommonDialect(dbName) if dbName.contains("postgres") => DatabaseType.Postgres
    case _ => DatabaseType.H2
  }

  def isPostgresDB(dialect: Dialect): Boolean = {
    getDatabaseType(dialect) match {
      case DatabaseType.Postgres => true
      case _ => false
    }
  }
}
