package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.v1.filter.WorkflowFilters
import com.xebialabs.xlrelease.api.v1.views.WorkflowOverview
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.WorkflowRepository
import com.xebialabs.xlrelease.repository.query.PageableQuery
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TAGS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{CategoryPersistence, PersistenceSupport, Utils}
import com.xebialabs.xlrelease.repository.sql.query.{SqlWorkflowFiltersQueryBuilder, WorkflowQueryBuilder}
import com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.JdbcTemplate

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlWorkflowRepository(permissionEnforcer: PermissionEnforcer,
                            roleService: RoleService,
                            categoryPersistence: CategoryPersistence,
                            implicit val dialect: Dialect,
                            implicit val jdbcTemplate: JdbcTemplate)
  extends WorkflowRepository
    with PersistenceSupport
    with Utils
    with Logging {

  @IsReadOnly
  @Timed
  override def findBy(workflowFilters: WorkflowFilters, pageable: Pageable): Page[WorkflowOverview] = {
    val builder = WorkflowQueryBuilder(dialect, namedTemplate).from(workflowFilters).asInstanceOf[SqlWorkflowFiltersQueryBuilder]
    if (!permissionEnforcer.isCurrentUserAdmin) {
      builder.withPermissions(Seq(VIEW_TEMPLATE), currentPrincipals(), currentRoleIds())
    }
    val pageableQuery: PageableQuery[WorkflowOverview] = builder.withPageable(pageable).build()
    val result = pageableQuery.execute()
    val content = result.getContent.asScala
    if (content.nonEmpty) {
      val fetchedReleases: Set[Integer] = content.map(_.getCiUid).toSet
      val releaseTags = findTags(fetchedReleases)
      val categories = findCategories(fetchedReleases)
      content.foreach { item =>
        item.setTags(releaseTags.getOrElse(item.getCiUid, Set[String]()).toList.asJava)
        item.setCategories(categories.getOrElse(item.getCiUid, Set[String]()).asJava)
      }
    }

    result
  }

  private val STMT_GET_TAGS_BY_RELEASE_UIDS =
    s"SELECT ${TAGS.CI_UID}, ${TAGS.VALUE} FROM ${TAGS.TABLE} WHERE ${TAGS.CI_UID} IN (:releaseUids)"

  private def findTags(releaseCiUids: Set[Integer]): Map[Int, Set[String]] = {
    logger.debug(s"Searching tags for releases with CiUids $releaseCiUids")
    val releaseTags = sqlQuery(
      STMT_GET_TAGS_BY_RELEASE_UIDS,
      params("releaseUids" -> releaseCiUids.asJava),
      (rs, _) => (rs.getInt(TAGS.CI_UID), rs.getString(TAGS.VALUE))
    ).toSet

    releaseTags.groupMap(_._1)(_._2)
  }

  private def findCategories(releaseCiUids: Set[Integer]): Map[Int, Set[String]] = {
    logger.debug(s"Searching categories for releases with CiUids $releaseCiUids")
    categoryPersistence.findActiveCategoryByReleaseCiUids(releaseCiUids)
  }

  private def currentPrincipals(): Iterable[String] =
    authenticationToPrincipals(getAuthentication).asScala

  private def currentRoleIds(): Iterable[String] =
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
}
