package com.xebialabs.xlrelease.security.sql.snapshots.service

import com.xebialabs.deployit.core.rest.api.SecurityResource
import com.xebialabs.deployit.security.{PermissionEditor, Role, RoleService}
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.sql.SqlFolderRepository
import com.xebialabs.xlrelease.repository.{Ids, SecuredCis}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.security.sql.snapshots.domain.{ContainerTeamsAndRolesForPrincipalAndPermission, RoleWithPrincipalsAndPermissions, RolesSnapshot, RolesSnapshotUnavailableError}
import com.xebialabs.xlrelease.security.sql.snapshots.repository.SqlRolesSnapshotRepository
import com.xebialabs.xlrelease.service.UserProfileService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
@EventListener
class SqlRolesSnapshotService @Autowired()(
                                            val sqlPermissionsSnapshotRepository: SqlRolesSnapshotRepository,
                                            val permissionChecker: PermissionChecker,
                                            val roleService: RoleService,
                                            val securityResource: SecurityResource,
                                            val permissionEditor: PermissionEditor,
                                            val securedCis: SecuredCis,
                                            val userProfileService: UserProfileService,
                                            val sqlFolderRepository: SqlFolderRepository
                                          ) extends PermissionsSnapshotService with Logging {

  @Subscribe
  def onGlobalRolesOrPermissionsUpdated(ev: GlobalRolesOrPermissionsUpdatedEvent): Unit = makeSnapshot()

  @Subscribe
  def onTeamsUpdated(ev: TeamsUpdatedEvent): Unit = makeSnapshot(Some(ev.containerId))

  override def makeSnapshot(containerId: Option[String] = None): Unit = {
    val containerIdName = containerId.map(Ids.getName)
    val parentContainerIdName = containerId.filter(!Ids.isRoot(_)).map(Ids.getParentId).filter(!Ids.isRoot(_)).map(Ids.getName)
    // do not save snapshots when containerId is a release inside a folder
    if (!containerId.exists(id => Ids.isReleaseId(id) && Ids.isInFolder(id))) {
      val containerSecurityUid = containerId.map(securedCis.getEffectiveSecuredCi(_).getSecurityUid).getOrElse(GLOBAL_SECURITY_ALIAS)
      val isInherited = containerId.exists(sqlFolderRepository.isFolderInherited)

      val snapshot = if (isInherited) {
        RolesSnapshot(new Date(), containerIdName, None, isInherited, parentContainerIdName)
      } else {
        val rolePermissionMultimap = permissionEditor.readPermissions(containerSecurityUid, false).asScala
        val roles: mutable.Seq[Role] = containerId.map(_ => roleService.readRoleAssignments(containerSecurityUid).asScala)
          .getOrElse(roleService.readRoleAssignments().asScala)
        RolesSnapshot(
          new Date(),
          containerIdName,
          Some(roles.map(r => RoleWithPrincipalsAndPermissions(
            r.getId,
            r.getName,
            r.getPrincipals.asScala.toSeq,
            r.getRoles.asScala.toSeq,
            rolePermissionMultimap.get(r).map(_.asScala.map(_.getPermissionName).toSeq).getOrElse(Seq())
          )).toSeq),
          isInherited = false,
          parentContainerIdName
        )
      }

      sqlPermissionsSnapshotRepository.saveSnapshot(snapshot)
    }
  }

  override def findSnapshotWithoutPermissionsCheck(containerId: Option[String], byDate: Date): Option[RolesSnapshot] = {
    sqlPermissionsSnapshotRepository.findSnapshot(containerId.map(Ids.getName), byDate)
  }

  override def findNextSnapshotWithoutPermissionsCheck(containerId: Option[String], date: Date): Option[RolesSnapshot] = {
    sqlPermissionsSnapshotRepository.findNextSnapshot(containerId.map(Ids.getName), date)
  }

  override def getMergedContainerSnapshotDataForReport(containerId: String, fromDate: Date, toDate: Date): (Seq[ContainerTeamsAndRolesForPrincipalAndPermission], Seq[RolesSnapshotUnavailableError]) = {
    mapDataForReport(Some(containerId), fromDate, toDate)(buildContainerTeamsAndRolesForPrincipalAndPermission)
  }

  private def buildContainerTeamsAndRolesForPrincipalAndPermission: (String, String, Seq[String]) => ContainerTeamsAndRolesForPrincipalAndPermission = {
    (principal: String, permission: String, roles: Seq[String]) =>
      ContainerTeamsAndRolesForPrincipalAndPermission(
        principal,
        Option(userProfileService.findByUsername(principal)).map(_.getFullName).getOrElse(principal), // this is using cache
        permission,
        roles
      )
  }

  private def getSnapshotSequence(
                                   containerId: Option[String],
                                   fromDate: Date,
                                   toDate: Date,
                                   errors: mutable.ListBuffer[RolesSnapshotUnavailableError]
                                 ): Iterable[(RolesSnapshot, Option[Date])] = {
    var snapshot: Option[RolesSnapshot] = findSnapshotWithoutPermissionsCheck(containerId, fromDate)
    if (snapshot.isEmpty) {
      snapshot = findNextSnapshotWithoutPermissionsCheck(containerId, fromDate)
      if (snapshot.isEmpty) {
        try {
          makeSnapshot(containerId)
          snapshot = findSnapshotWithoutPermissionsCheck(containerId, new Date())
        } catch {
          // log and kill exception
          case e: Exception => logger.error(
            s"Unable to find any snapshot for folder [${containerId.getOrElse("/")}] " +
              s"and was unable to create new snapshot", e)
        }
      }
      errors += RolesSnapshotUnavailableError(containerId, fromDate, snapshot.map(_.snapshotDate))
    }

    def fetchNextSnapshot(): Option[RolesSnapshot] = {
      snapshot
        .map(_.snapshotDate)
        .flatMap(findNextSnapshotWithoutPermissionsCheck(containerId, _))
    }

    var nextSnapshot: Option[RolesSnapshot] = fetchNextSnapshot()

    new Iterable[(RolesSnapshot, Option[Date])]() {
      override def iterator: Iterator[(RolesSnapshot, Option[Date])] = {
        new Iterator[(RolesSnapshot, Option[Date])]() {
          override def hasNext: Boolean = snapshot.isDefined

          override def next(): (RolesSnapshot, Option[Date]) = {
            val result = snapshot.map(s => (s, nextSnapshot.map(_.snapshotDate)))
            if (nextSnapshot.isEmpty || (toDate != null && toDate.equals(fromDate))) {
              snapshot = None // this is last snapshot or range is not requested
              nextSnapshot = None
            } else if (toDate != null && toDate.getTime < nextSnapshot.get.snapshotDate.getTime) {
              // toDate is set and is earlier than next snapshot
              snapshot = None
              nextSnapshot = None
            } else {
              snapshot = nextSnapshot
              nextSnapshot = fetchNextSnapshot()
            }
            result.get
          }
        }
      }
    }
  }

  private def mapDataForReportInto(
                                    containerId: Option[String],
                                    fromDate: Date,
                                    toDate: Date,
                                    resultData: mutable.Map[(String, String), mutable.LinkedHashSet[String]],
                                    errors: mutable.ListBuffer[RolesSnapshotUnavailableError]
                                  ): Unit = {
    getSnapshotSequence(containerId, fromDate, toDate, errors).foreach {
      case (snapshot, maybeNextSnapshotDate) =>
        if (snapshot.isInherited) {
          if (snapshot.parentContainerIdName.isDefined) {
            mapDataForReportInto(snapshot.parentContainerIdName, snapshot.snapshotDate, maybeNextSnapshotDate.orNull, resultData, errors)
          } else {
            logger.error(s"Strange snapshot found for [${containerId.getOrElse("/")}] which has isInherited but does not have parent, skipping")
          }
        } else {
          if (containerId.isDefined) {
            // this is folder snapshot, we need to walk through global snapshots to resolve folder roles
            (getSnapshotSequence(None, snapshot.snapshotDate, maybeNextSnapshotDate.orNull, errors) match {
              case empty if empty.isEmpty => Seq((RolesSnapshot.empty, None))
              case nonEmpty => nonEmpty
            }).foreach {
              case (globalSnapshot, _) =>
                populateResultData(resultData, containerId, snapshot, Some(globalSnapshot))
            }
          } else {
            // this is global snapshot
            populateResultData(resultData, containerId, snapshot)
          }
        }
    }
  }

  private def mapDataForReport[A](containerId: Option[String], fromDate: Date, toDate: Date)
                                 (mapF: (String, String, Seq[String]) => A): (Seq[A], Seq[RolesSnapshotUnavailableError]) = {
    val resultData = mutable.Map[(String, String), mutable.LinkedHashSet[String]]()
    val errors = mutable.ListBuffer[RolesSnapshotUnavailableError]()
    mapDataForReportInto(containerId, fromDate, toDate, resultData, errors)
    (
      resultData.toSeq.map {
        case ((principal, permission), roles) => mapF(principal, permission, roles.toSeq)
      },
      errors.toSeq
    )
  }

  private type ResultDataMap = mutable.Map[(String, String), mutable.LinkedHashSet[String]]

  private def populateResultData(resultData: ResultDataMap,
                                 containerId: Option[String],
                                 snapshot: RolesSnapshot,
                                 globalSnapshot: Option[RolesSnapshot] = None): Unit = {

    val globalRoleNameToPrincipals = globalSnapshot
      .flatMap(_.rolesAndPermissions)
      .foldLeft(Map.empty[String, Set[String]]) {
        case (acc, rolesSnapshot) =>
          var result = acc
          rolesSnapshot.foreach { roleWithPrincipalsAndPermissions =>
            val roleName = roleWithPrincipalsAndPermissions.name
            result += roleName -> (result.getOrElse(roleName, Set.empty) ++ roleWithPrincipalsAndPermissions.principals.toSet)
          }
          result
      }

    snapshot.rolesAndPermissions.foreach(_.foreach(role => {
      role.permissions.foreach(permission => {
        val principals = role.principals.toSet
        principals.foreach(principal => {
          resultData.getOrElseUpdate((principal, permission), mutable.LinkedHashSet()).add(role.name)
        })
        if (containerId.isDefined) {
          val globalPrincipals = role.roleIds.flatMap(globalRoleNameToPrincipals.getOrElse(_, Set.empty)).toSet
          role.roleIds.map(roleName => {
            globalRoleNameToPrincipals.getOrElse(roleName, Set.empty).foreach(principal => {
              resultData.getOrElseUpdate((principal, permission), mutable.LinkedHashSet()).add(s"${role.name} ($roleName)")
            })
          })
        }
      })
    }))
  }

}
