package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES
import com.xebialabs.xlrelease.api.internal.ReleaseServerUrlDecorator.SERVER_URL
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, EffectiveSecurity, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.api.v1.forms.ReleaseOrderMode.title
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, ReleasesFilters}
import com.xebialabs.xlrelease.builder.ReleaseBuilder
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.domain.status.ReleaseStatus._
import com.xebialabs.xlrelease.domain.status.{FlagStatus, ReleaseStatus}
import com.xebialabs.xlrelease.domain.utils.ReleaseIdInDatabase
import com.xebialabs.xlrelease.domain.{OriginTemplateData, Release, ReleaseKind}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.ReleaseExtensionsRepository.FindByNameAndCiUid
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.data.OriginTemplateDataRow
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, ReleasesSqlBuilder}
import com.xebialabs.xlrelease.repository.sql.{DeserializationSupport, SqlRepositoryAdapter}
import com.xebialabs.xlrelease.risk.domain.progress.ReleaseProgress
import com.xebialabs.xlrelease.search._
import com.xebialabs.xlrelease.security.XLReleasePermissions.{AUDIT_ALL, VIEW_RELEASE, VIEW_TEMPLATE}
import com.xebialabs.xlrelease.utils.Diff
import com.xebialabs.xlrelease.views._
import grizzled.slf4j.Logging

import java.sql.Timestamp
import java.util.{Date, List => JList}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class SqlReleaseSearchService(releasePersistence: ReleasePersistence,
                              archivingService: ArchivingService,
                              val repositoryAdapter: SqlRepositoryAdapter,
                              val teamService: TeamService,
                              decoratorService: InternalMetadataDecoratorService,
                              implicit val sqlDialect: Dialect,
                              implicit val permissionEnforcer: PermissionEnforcer,
                              val roleService: RoleService,
                              val securedCis: SecuredCis,
                              releaseExtensionsRepository: ReleaseExtensionsRepository)
  extends ReleaseSearchService
    with Logging
    with DeserializationSupport {

  @Timed
  override def count(releasesFilters: ReleasesFilters): ReleaseCountResults = {
    releasesFilters.setOrderBy(null)

    val fromArchive = archivingService.countReleasesByStatus(releasesFilters).view.mapValues(Integer.valueOf).toMap
    val (current, archived) = if (releasesFilters.withOnlyArchived()) {
      Map.empty[ReleaseStatus, Integer] -> fromArchive
    } else {
      val principals = currentPrincipals()
      val roleIds = currentRoleIds()
      val statusQuery = SqlReleasesFilterSupport.sqlBuilderByFilters(releasesFilters, principals, roleIds)
        .countByStatus()
      releasePersistence.countReleasesByStatus(statusQuery.build()).view.mapValues(Integer.valueOf) -> fromArchive
    }
    new ReleaseCountResults(current.toMap.asJava, archived.asJava)
  }

  @Timed
  override def getReleaseDateRange(releasesFilters: ReleasesFilters): ReleaseDateRangeResults = {
    releasesFilters.setOrderBy(null)
    if (releasesFilters.withOnlyArchived()) {
      throw new IllegalStateException("Date range does not work with archived");
    }
    val principals = currentPrincipals()
    val roleIds = currentRoleIds()
    val dateRangeQuery = SqlReleasesFilterSupport.sqlBuilderByFilters(releasesFilters, principals, roleIds)
      .dateRange()

    val dateRange = releasePersistence.dateRange(dateRangeQuery.build())

    new ReleaseDateRangeResults(dateRange._1, dateRange._2)
  }

  @Timed
  override def fullSearch(releasesFilters: ReleasesFilters, currentPage: Page, archivePage: Page): ReleaseFullSearchResult = {
    val getIds: Page => Seq[String] = searchIds(releasesFilters)
    val archived = new ReleaseSearchResult(searchInArchive(releasesFilters, archivePage.resultsPerPage, archivePage.offset).asJava, archivePage.page)
    val live = if (releasesFilters.withOnlyArchived()) {
      ReleaseSearchResult.empty(currentPage.page)
    } else {
      new ReleaseSearchResult(queryReleases(getIds, currentPage.page, currentPage.resultsPerPage).asJava, currentPage.page)
    }
    new ReleaseFullSearchResult(live, archived)
  }

  @Timed
  override def search(releasesFilters: ReleasesFilters, page: Long, numberByPage: Long, pageIsOffset: Boolean,
                      depth: Int, filterOnPermissions: Boolean, includeOriginTemplateData: Boolean): ReleaseSearchResult = {
    val results = {
      if (releasesFilters.withOnlyArchived()) {
        searchInArchive(releasesFilters, numberByPage, if (pageIsOffset) page else numberByPage * page)
      } else {
        queryReleases(searchIds(releasesFilters), page, numberByPage, includeOriginTemplateData, pageIsOffset)
      }
    }
    toSearchResult(page, results)
  }

  private val mappedFlagStatus = FlagStatus.values().map(v => (v.getRisk, v)).toMap

  @Timed
  override def releasesOverview(releasesFilters: ReleasesFilters, pageNum: Long, numberByPage: Long): ReleaseOverviewResultsView = {
    val releases = if (releasesFilters.withOnlyArchived()) {
      archivingService.overviewReleases(releasesFilters, numberByPage, pageNum * numberByPage)
    } else {
      val rows = releasePersistence.releasesOverview(releasesFilters, pageNum, numberByPage, currentPrincipals(), currentRoleIds(), permissionEnforcer)
      val releaseUids = rows.collect { case row => row.ciUid }
      val query = FindByNameAndCiUid(ReleaseProgress.PROGRESS_PREFIX, releaseUids)
      val progressListByRelease = releaseExtensionsRepository.find(query).groupBy(e => Ids.releaseIdFrom(e.getId))
      rows.map { row =>
        val overview = new ReleaseOverview()
        overview.setType(Type.valueOf(classOf[Release]).toString)
        overview.setId(row.releaseId)
        overview.setTitle(row.title)
        overview.setStartDate(row.startDate)
        overview.setEndDate(row.endDate)
        overview.setStatus(ReleaseStatus.valueOf(row.status.toUpperCase))
        overview.setKind(ReleaseKind.valueOf(row.kind.toUpperCase))
        overview.setPlannedDuration(row.plannedDuration.toLong)
        overview.setArchived(false)
        overview.setRiskScore(row.riskScore)
        val status = mappedFlagStatus.getOrElse(row.realFlagStatus, FlagStatus.OK)
        val flag = new Flag(status)
        overview.setFlag(flag)
        for {
          progressList <- progressListByRelease.get(row.releaseId)
          extension <- progressList.headOption
        } {
          val progress = extension.asInstanceOf[ReleaseProgress]
          val totalRemainingTasks = progress.getTotalRemainingTasks
          val totalTasks = progress.getTotalTasks
          val tasksFinished = totalTasks - totalRemainingTasks
          val calculatedProgress = if (tasksFinished == 0) 0 else Math.round((tasksFinished * 100) / totalTasks)
          overview.setTotalRemainingTasks(totalRemainingTasks)
          overview.setTotalTasks(totalTasks)
          overview.setCalculatedProgressPercent(calculatedProgress)
        }
        overview.setCurrentPhase(row.currentPhaseTitle)
        overview
      }.asJava
    }
    new ReleaseOverviewResultsView(pageNum + 1, releases) //  `pageNum + 1` is here because UI expects it
  }

  @Timed
  override def templatesOverview(templateFilters: TemplateFilters, pageNum: Long, numberByPage: Long): TemplateOverviewResultsView = {
    val rows = releasePersistence.templatesOverview(templateFilters, pageNum, numberByPage, currentPrincipals(), currentRoleIds(), permissionEnforcer)
    val templates = rows.map { row =>
      val overview = new TemplateOverview()
      overview.setType(Type.valueOf(classOf[Release]).toString)
      overview.setId(row.releaseId)
      overview.setTitle(row.title)
      overview.setPlannedDuration(row.plannedDuration.toLong)
      overview.setSecurityUid(row.securityUid)
      overview.setStatus(row.status.toUpperCase)
      overview.setKind(row.kind.toUpperCase)
      overview.set$scmTraceabilityDataId(row.scmDataUid)
      overview
    }

    decorateWithEffectiveSecurity(templates)
    new TemplateOverviewResultsView(pageNum + 1, templates.asJava)
  }

  private def decorateWithEffectiveSecurity(templates: List[TemplateOverview]): Unit = {
    // for now reuse existing (slowish) logic to decorate templates with effective security
    val templateMap: Map[String, TemplateOverview] = templates.map(t => t.getId -> t).toMap
    // decorating rows with effective security is slow (just fetching bare data is ~60ms buf if we also add effective security it jumps to >200ms)
    val cis = templates.map { t =>
      val r = ReleaseBuilder.newTemplate().withId(t.getId).build()
      r.set$securedCi(t.getSecurityUid)
      r
    }.asJava
    decoratorService.decorate(cis, List(EFFECTIVE_SECURITY).asJava)
    cis.forEach { ci =>
      templateMap.get(ci.getId).foreach { t =>
        val effectiveSecurity = ci.get$metadata().get(EFFECTIVE_SECURITY).asInstanceOf[EffectiveSecurity]
        t.setSecurity(effectiveSecurity)
      }
    }
  }

  @Timed
  override def searchReleasesByReleaseIds(releaseIds: Seq[String]): Seq[Release] = {
    queryReleasesByIds(releaseIds)
  }

  @Timed
  override def searchAllIdsWithoutPermissionsCheck(filters: Seq[ReportFilter],
                                                   order: Option[ReleaseOrderMode],
                                                   page: Option[Long] = None,
                                                   numberByPage: Option[Long] = None): Seq[ReleaseIdInDatabase] = {
    val nonArchiveSqlBuilder: ReleasesSqlBuilder = createReleaseSqlBuilder(filters)
    nonArchiveSqlBuilder.selectShortReleaseIdAndOrderCriterion(order)
    order.foreach(o => nonArchiveSqlBuilder.orderBy(o, Some(true)))

    val limit = if (page.isDefined && numberByPage.isDefined) {
      Some((page.get + 1) * numberByPage.get)
    } else {
      None
    }

    limit.foreach(nonArchiveSqlBuilder.limit)

    val nonArchived = releasePersistence.findShortReleaseIdsWithFolderNameAndOrderCriterionByQuery(
      order, nonArchiveSqlBuilder.build())

    val archived = archivingService.findShortReleaseIdsWithFolderNameAndOrderCriterion(filters, order, limit)

    // let's merge two data sets and sort them
    val diff = Diff.applyWithKeyMapping(archived, nonArchived)(t => t._1.folderlessReleaseId())
    val both = (diff.newEntries.view.mapValues(t => ReleaseIdInDatabase(t._1, inArchive = false, inNonArchive = true) -> t._2) ++
      diff.deletedEntries.view.mapValues(t => ReleaseIdInDatabase(t._1, inArchive = true, inNonArchive = false) -> t._2) ++
      diff.updatedEntries.view.mapValues(t => ReleaseIdInDatabase(t._1._1.combineWith(t._2._1), inArchive = true, inNonArchive = true) -> t._2._2)).toMap.values.toBuffer
    val sorted = order match {
      case None => both
      case Some(o) => both.sortWith(o match {
        case ReleaseOrderMode.end_date |
             ReleaseOrderMode.start_date => (a, b) => a._2.asInstanceOf[Timestamp].compareTo(b._2.asInstanceOf[Timestamp]) < 0
        case ReleaseOrderMode.title => (a, b) => a._2.asInstanceOf[String].compareTo(b._2.asInstanceOf[String]) < 0
        case _ => throw new IllegalStateException(s"Unable to order by [$o]")
      })
    }
    if (limit.isDefined) {
      sorted.map(_._1).slice((limit.get - numberByPage.get).toInt, limit.get.toInt).toSeq
    } else {
      sorted.map(_._1).toSeq
    }
  }

  @Timed
  override def searchTemplates(templateFilters: TemplateFilters, page: Long, numberByPage: Long, depth: Int, enforcePermission: Boolean = true): ReleaseSearchResult = {
    logger.debug(s"Searching for templates by filters")

    val query = SqlTemplatesFilterSupport.sqlBuilderByFilters(templateFilters, currentPrincipals(), currentRoleIds(), enforcePermission)
      .selectReleaseId()

    def searchIds(page: Page): Seq[String] = {
      val sqlWithParams = query.withPage(page).build()
      logger.debug(s"Searching for a batch of templates by SQL query: ${sqlWithParams._1}")
      releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
    }

    val templates = queryReleases(searchIds, page, numberByPage)

    toSearchResult(page, templates)
  }

  @Timed
  override def searchReleasesByTitle(releaseTitle: String, releaseKind: ReleaseKind, page: Long, numberByPage: Long, depth: Int): JList[Release] = {
    logger.debug(s"Searching for releases by title $releaseTitle")

    val query = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withTitle(releaseTitle)
      .withKind(releaseKind)

    if (!permissionEnforcer.isCurrentUserAdmin) {
      query.withPermission(VIEW_RELEASE, currentPrincipals(), currentRoleIds())
    }

    def getIds(page: Page): Seq[String] = {
      val sqlWithParams = query.withPage(page).build()
      releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
    }

    queryReleases(getIds, page, numberByPage).asJava
  }

  @Timed
  override def findAllReleaseIdsAndTitlesByStatus(statuses: ReleaseStatus*): Map[String, String] = {
    val sqlBuilder = new ReleasesSqlBuilder()
      .selectReleaseIdAndTitle()
      .withKind(ReleaseKind.RELEASE)
      .withOneOfStatuses(statuses)
    if (!permissionEnforcer.isCurrentUserAdmin) {
      sqlBuilder.withPermission(VIEW_RELEASE, currentPrincipals(), currentRoleIds())
    }
    val sqlWithParams = sqlBuilder.build()
    releasePersistence.findReleaseIdsAndTitlesByQuery(sqlWithParams).toMap
  }

  @Timed
  override def findAllTemplateIdsAndTitles(page: Long,
                                           numberByPage: Long,
                                           folderId: Option[String],
                                           permission: Permission = VIEW_TEMPLATE,
                                           matchTemplate: Option[String]): PlanItemSearchResult = {
    val sqlBuilder = new ReleasesSqlBuilder()
      .selectReleaseIdAndTitle()
      .withKind(ReleaseKind.RELEASE)
      .withOneOfStatuses(Seq(TEMPLATE))
      .orderBy(title)
      .withPage(Page(page.toInt, numberByPage.toInt, 0))
    folderId.filter(!_.isEmpty).foreach(sqlBuilder.withFolder)
    matchTemplate.filter(!_.isEmpty).foreach(sqlBuilder.withTitleLike)
    if (!permissionEnforcer.isCurrentUserAdmin && (!permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL) || permission != VIEW_TEMPLATE)) {
      sqlBuilder.withPermission(permission, currentPrincipals(), currentRoleIds())
    }
    val idToTitle = releasePersistence.findReleaseIdsAndTitlesByQuery(sqlBuilder.build()).toMap
    val searchResult = new PlanItemSearchResult()
    searchResult.setPage(page + 1)
    searchResult.setItems(idToTitle.asJava)
    searchResult
  }

  @Timed
  override def findTemplateTitleById(permission: Permission = VIEW_TEMPLATE,
                                     templateId: String): PlanItemSearchResult = {
    val sqlBuilder = new ReleasesSqlBuilder()
      .selectReleaseIdAndTitle()
      .withReleaseIds(Seq(templateId))
      .withOneOfStatuses(Seq(TEMPLATE))
    if (!permissionEnforcer.isCurrentUserAdmin && (!permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL) || permission != VIEW_TEMPLATE)) {
      sqlBuilder.withPermission(permission, currentPrincipals(), currentRoleIds())
    }
    val idToTitle = releasePersistence.findReleaseIdsAndTitlesByQuery(sqlBuilder.build()).toMap
    val searchResult = new PlanItemSearchResult()
    searchResult.setItems(idToTitle.asJava)
    searchResult
  }

  @Timed
  override def findNonArchivedExpiredReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    val sqlBuilder = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(INACTIVE_STATUSES)
      .withEndDateBefore(date)
      .withPreArchived(false)
      .withPage(new Page(0, pageSize.longValue(), 0))
    releasePersistence.findReleaseIdsByQuery(sqlBuilder.build()).toSeq
  }

  @Timed
  override def getTitle(releaseId: String): Option[String] = {
    releasePersistence.findReleaseTitle(releaseId).orElse {
      Try(archivingService.getReleaseTitle(releaseId)).toOption
    }
  }

  @Timed
  override def findAllTemplateIds: JList[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(Seq(TEMPLATE))
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).asJava
  }

  @Timed
  override def findAllActiveReleaseIds: JList[String] = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(RECOVERABLE_STATUSES)
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).asJava
  }

  @Timed
  override def findAllPendingReleaseIds: JList[String] = {
    logger.debug(s"Searching for all pending releases")
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withOneOfStatuses(Seq(PLANNED))
      .withAutoStart()
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).asJava
  }

  private def queryReleases(searchIds: Page => Seq[String],
                            page: Long, numberByPage: Long, includeOriginTemplateData: Boolean = false, pageIsOffset: Boolean = false): Seq[Release] = {
    queryReleasesByIds(searchIds(Page(page, numberByPage, 0, pageIsOffset)), includeOriginTemplateData)
  }

  private def queryReleasesByIds(releaseIds: Seq[String], includeOriginTemplateData: Boolean = false): Seq[Release] = {
    val releases = if (releaseIds.nonEmpty) {
      if (!includeOriginTemplateData) {
        findReleasesByIdsInOrder(releaseIds)
      } else {
        findReleasesByIdsInOrderAndDecorateWithTemplateData(releaseIds)
      }
    } else {
      Seq.empty[Release]
    }

    // REL-7718: silently ignore decoration errors, i.e. if release was being archived (deleted from live db)
    val decoratorCache = new DecoratorsCache()
    releases.map { release =>
      for {
        _ <- Try(teamService.decorateWithEffectiveTeams(release, decoratorCache))
        _ <- Try(decoratorService.decorate(release, Seq(GLOBAL_AND_FOLDER_VARIABLES, EFFECTIVE_SECURITY, SERVER_URL).asJava, decoratorCache)) //FIXME use cis method
      } yield release
    }.flatMap {
      case Failure(err) =>
        logger.warn(err)
        List.empty
      case Success(release) =>
        List(release)
    }
  }

  private def findReleasesByIdsInOrder(releaseIds: Seq[String]): Seq[Release] = {
    val sqlWithParams = new ReleasesSqlBuilder().selectReleaseData()
      .withReleaseIds(releaseIds).build()
    val releases = fetchReleaseDataDeserializeAndDecorate(sqlWithParams)
    orderReleasesByIds(releases, releaseIds)
  }

  private def fetchReleaseDataDeserializeAndDecorate(sqlWithParams: SqlWithParameters): Seq[Release] = {
    releasePersistence.findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .map(releaseExtensionsRepository.decorate)
  }

  private def findReleasesByIdsInOrderAndDecorateWithTemplateData(releaseIds: Seq[String]): Seq[Release] = {
    val sqlWithParams = new ReleasesSqlBuilder().selectReleaseDataAndTemplateData()
      .withReleaseIds(releaseIds).build()
    val releases = fetchReleaseDataWithTemplateDataDeserializeAndDecorate(sqlWithParams)
    orderReleasesByIds(releases, releaseIds)
  }

  private def fetchReleaseDataWithTemplateDataDeserializeAndDecorate(sqlWithParams: SqlWithParameters): Seq[Release] =
    releasePersistence.findReleaseDataWithTemplateDataByQuery(sqlWithParams).flatMap { releaseAndTemplateDataRow =>
      val (releaseRow, originTemplateDataRow) = releaseAndTemplateDataRow
      tryDeserializeRelease(releaseRow)
        .map(releaseExtensionsRepository.decorate)
        .map(decorateWithTemplateDataAsReleaseExtension(originTemplateDataRow))
    }

  private def orderReleasesByIds(releases: Seq[Release], ids: Seq[String]): Seq[Release] =
    releases.sortBy(r => ids.indexOf(r.getId))

  private def toSearchResult(page: Long, releases: Seq[Release]): ReleaseSearchResult = {
    new ReleaseSearchResult(releases.asJava, page + 1)
  }

  private def searchIds(releasesFilters: ReleasesFilters): Page => Seq[String] = {
    if (releasesFilters.getOrderBy == null) {
      releasesFilters.setOrderBy(ReleaseOrderMode.risk)
    }
    val query = SqlReleasesFilterSupport.sqlBuilderByFilters(releasesFilters, currentPrincipals(), currentRoleIds())
      .selectReleaseId()

    page => {
      val sqlWithParams = query.withPage(page).build()
      logger.debug(s"Searching for a batch of releases by SQL query: ${sqlWithParams._1}")
      releasePersistence.findReleaseIdsByQuery(sqlWithParams).toSeq
    }
  }

  private def searchInArchive(releasesFilters: ReleasesFilters, numberByPage: Long, offset: Long): Seq[Release] = {
    val releases = archivingService.searchReleases(releasesFilters, numberByPage, offset)
    decoratorService.decorate(releases, List(GLOBAL_AND_FOLDER_VARIABLES, EFFECTIVE_SECURITY, SERVER_URL).asJava, new DecoratorsCache())
    releases.asScala.toSeq
  }

  @Timed
  override def existsByName(candidateName: String): Boolean = {
    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseId()
      .withReleaseName(candidateName)
      .withPage(Page(0, 1, 0))
      .build()
    releasePersistence.findReleaseIdsByQuery(sqlWithParams).nonEmpty
  }

  @Timed
  def countReleases(filters: Seq[ReportFilter]): ReleaseCountResults = {
    val statusQuery: ReleasesSqlBuilder = createReleaseSqlBuilder(filters)
    statusQuery.countByStatus()
    val current = releasePersistence.countReleasesByStatus(statusQuery.build()).view.mapValues(Integer.valueOf).toMap
    val archived = archivingService.countReleases(filters).view.mapValues(Integer.valueOf).toMap
    new ReleaseCountResults(current.asJava, archived.asJava)
  }

  private def createReleaseSqlBuilder(filters: Seq[ReportFilter]): ReleasesSqlBuilder = {
    val statusQuery = new ReleasesSqlBuilder()
    filters.foreach(_.apply(statusQuery))
    statusQuery.withoutStatuses(Seq(ReleaseStatus.PLANNED, ReleaseStatus.TEMPLATE))
    statusQuery
  }

  private def currentPrincipals(): Iterable[String] =
    authenticationToPrincipals(getAuthentication).asScala

  private def currentRoleIds(): Iterable[String] =
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)

  private def decorateWithTemplateDataAsReleaseExtension(originTemplateDataRow: OriginTemplateDataRow): Release => Release =
    (release: Release) => {
      if (isNotEmpty(originTemplateDataRow.originTemplateId) && isNotEmpty(originTemplateDataRow.originTemplateTitle)) {
        val extension = OriginTemplateData(
          releaseId = release.getId,
          templateId = originTemplateDataRow.originTemplateId,
          templateTitle = originTemplateDataRow.originTemplateTitle
        )
        release.setExtensions((release.getExtensions.asScala :+ extension).asJava)
      }
      release
    }
}
