package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.UserProfileRepository
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.security.sql.SecurityUserCacheConfigurationCondition
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.views.users.UserFilters
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.cache.CacheManager
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable, Caching}
import org.springframework.context.annotation.Conditional
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.stereotype.Component

import java.lang
import java.util.{Date, List => JList}
import scala.util.{Failure, Try}

@CacheConfig(cacheManager = SECURITY_USER_CACHE_MANAGER)
class CachingSqlUserProfileRepositoryWrapper(repository: UserProfileRepository, evicter: UserProfileCacheEvicter) extends UserProfileRepository with Logging {

  override def create(userProfile: UserProfile): Unit = {
    repository.create(userProfile)
    evicter.onUserProfileCreated(DistributedUserProfileCreated(userProfile.getCanonicalId))
  }

  override def update(userProfile: JList[UserProfile]): List[Try[UserProfile]] = {
    val tryList = repository.update(userProfile)

    tryList.filter(_.isSuccess).map(_.get.getCanonicalId) match {
      case usernames: List[String] if usernames.nonEmpty => evicter.onAtLeastOneUserProfileUpdated(DistributedAtLeastOneUserProfileUpdated(usernames))
      case _ => ()
    }

    tryList.find(_.isFailure) match {
      case Some(Failure(ex)) => throw ex
      case _ => tryList
    }
  }

  override def updateLastActive(canonicalId: String, lastActive: Date): Boolean = repository.updateLastActive(canonicalId, lastActive)

  override def updateLastActiveBatch(entries: Map[String, Date]): Int = repository.updateLastActiveBatch(entries)

  override def delete(userProfileId: String): Unit = {
    repository.delete(userProfileId)
    evicter.onUserProfileDeleted(DistributedUserProfileDeleted(userProfileId))
  }

  @Cacheable(cacheNames = Array(SECURITY_USER_PROFILE), key = "#canonicalId", unless = "#result.isEmpty")
  override def findById(canonicalId: String): Option[UserProfile] = repository.findById(canonicalId)

  @Cacheable(cacheNames = Array(SECURITY_ALL_USER_PROFILES))
  override def findAll(fullProfile: Boolean): JList[UserProfile] = repository.findAll(fullProfile)

  override def exists(canonicalId: String): Boolean = repository.exists(canonicalId)

  override def customSearch(email: String, fullName: String, loginAllowed: lang.Boolean, lastActiveAfter: Date, lastActiveBefore: Date,
                            page: Option[Long], resultsPerPage: Option[Long]): JList[UserProfile] =
    repository.customSearch(email, fullName, loginAllowed, lastActiveAfter, lastActiveBefore, page, resultsPerPage)

  override def countUserWithLoginAllowed(): Int = repository.countUserWithLoginAllowed()

  override def searchUserProfiles(userFilters: UserFilters, pageable: Pageable): Page[UserProfile] = repository.searchUserProfiles(userFilters, pageable)
}

case class DistributedUserProfileCreated(canonicalId: String, publish: Boolean = true) extends DistributedXLReleaseEvent
case class DistributedAtLeastOneUserProfileUpdated(usernames: List[String], publish: Boolean = true) extends DistributedXLReleaseEvent
case class DistributedUserProfileDeleted(canonicalId: String, publish: Boolean = true) extends DistributedXLReleaseEvent

@Component
@EventListener
@Conditional(value = Array(classOf[SecurityUserCacheConfigurationCondition]))
@CacheConfig(cacheManager = SECURITY_USER_CACHE_MANAGER)
class UserProfileCacheEvicter(broadcastService: BroadcastService, @Qualifier(SECURITY_USER_CACHE_MANAGER) cacheManager: CacheManager) {

  @AsyncSubscribe
  @CacheEvict(cacheNames = Array(SECURITY_ALL_USER_PROFILES), allEntries = true)
  def onUserProfileCreated(event: DistributedUserProfileCreated): Unit = {
    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }

  @AsyncSubscribe
  @CacheEvict(cacheNames = Array(SECURITY_ALL_USER_PROFILES), allEntries = true)
  def onAtLeastOneUserProfileUpdated(event: DistributedAtLeastOneUserProfileUpdated): Unit = {
    val cache = cacheManager.getCache(SECURITY_USER_PROFILE)
    event.usernames.foreach(cache.evict)

    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }

  @AsyncSubscribe
  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_USER_PROFILE), key = "#event.canonicalId"),
    new CacheEvict(cacheNames = Array(SECURITY_ALL_USER_PROFILES), allEntries = true)
  ))
  def onUserProfileDeleted(event: DistributedUserProfileDeleted): Unit = {
    if (event.publish) {
      broadcastService.broadcast(event.copy(publish = false), false)
    }
  }
}
