package com.xebialabs.xlrelease.repository.sql

import com.google.common.base.Preconditions.checkState
import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.repository.sql.WorkflowTaskValidator._

private[sql] object WorkflowTaskValidator {
  private lazy val DEFAULT_MAX_TEMPLATE_TASK_COUNT = XlrConfig.getInstance.features.workflows.limits.maxTemplateTasks
  private lazy val DEFAULT_MAX_EXECUTION_TASK_COUNT = XlrConfig.getInstance.features.workflows.limits.maxExecutionTasks
}

private[sql] trait WorkflowTaskValidator {

  def validateNoLockTaskPresent(task: Task): Unit = {
    if (task.isLocked) {
      throw new IncorrectArgumentException("Lock tasks are not supported for workflows")
    }
  }

  def validateNoLockTaskPresent(tasks: Seq[Task]): Unit = {
    val hasLockTasks = tasks.exists(_.isLocked)
    if (hasLockTasks) {
      throw new IncorrectArgumentException("Lock tasks are not supported for workflows")
    }
  }

  def validateNoUnsupportedTaskPresent(task: Task): Unit = {
    if (!task.isSupportedInWorkflow) {
      throw new IncorrectArgumentException(s"Task type [${task.getTaskType.toString}] is not supported for workflow")
    }
  }

  def validateNoUnsupportedTaskPresent(tasks: Seq[Task]): Unit = {
    val unsupportedTasks = tasks.filter(!_.isSupportedInWorkflow)
    if (unsupportedTasks.nonEmpty) {
      val taskTypes = unsupportedTasks.map(_.getTaskType.toString).mkString(", ")
      throw new IncorrectArgumentException(s"Found unsupported task types for workflow: $taskTypes")
    }
  }

  def validateWorkflowTaskCount(release: Release): Unit = {
    if (release.isWorkflow) {
      if (release.isTemplate) {
        val message = s"Workflow templates cannot exceed $DEFAULT_MAX_TEMPLATE_TASK_COUNT tasks"
        checkState(release.getAllTasks.size() <= DEFAULT_MAX_TEMPLATE_TASK_COUNT, message.asInstanceOf[Object])
      } else {
        val message = s"Workflow executions cannot exceed $DEFAULT_MAX_EXECUTION_TASK_COUNT tasks"
        checkState(release.getAllTasks.size() <= DEFAULT_MAX_EXECUTION_TASK_COUNT, message.asInstanceOf[Object])
      }
    }
  }
}
