package com.xebialabs.xlrelease.repository.sql.security

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.utils.IdUtils
import com.xebialabs.xlrelease.repository.SecurityRepository.TeamId
import com.xebialabs.xlrelease.repository.{SecuredCis, SecurityRepository, TeamArrayPropertyChange}
import com.xebialabs.xlrelease.security.entities._
import com.xebialabs.xlrelease.security.sql.db.Ids
import com.xebialabs.xlrelease.security.{XlRolePermissionRepository, XlRolePrincipalRepository, XlRoleRepository, XlRoleRoleRepository}

import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters.RichOptional


@IsTransactional
class SqlSecurityRepository(roleRepository: XlRoleRepository,
                            roleRolesRepository: XlRoleRoleRepository,
                            rolePrincipalsRepository: XlRolePrincipalRepository,
                            rolePermissionsRepository: XlRolePermissionRepository,
                            securedCis: SecuredCis
                           ) extends SecurityRepository {

  override def getTeam(teamId: TeamId): Team = {
    val xlRole = findRequiredTeam(teamId)
    val team: Team = getTeamAttributes(xlRole)
    team
  }

  override def createTeam(containerId: TeamId,
                          teamName: TeamId,
                          roles: Set[String] = Set.empty,
                          principals: Set[String] = Set.empty,
                          permissions: Set[String] = Set.empty): TeamId = {
    val containerCiId = getContainerCiId(containerId)
    val team = findOptionalXlRoleByName(containerCiId, teamName) match {
      case Some(xlRole) => xlRole
      case None =>
        val roleId: String = IdUtils.getUniqueId("Team", "").stripPrefix("/")
        val xlRole = new XlRole(roleId, teamName, containerCiId)
        roleRepository.save(xlRole)
    }
    addRolesByNameToTeam(team, roles)
    addPrincipalsToTeam(team, principals)
    addPermissionsToTeam(team, permissions)
    team.getId
  }

  override def updateTeam(teamId: TeamId,
                          roles: TeamArrayPropertyChange,
                          principals: TeamArrayPropertyChange,
                          permissions: TeamArrayPropertyChange): TeamId = {
    val team = findRequiredTeam(teamId)

    addRolesByNameToTeam(team, roles.add)
    addPrincipalsToTeam(team, principals.add)
    addPermissionsToTeam(team, permissions.add)
    removeRolesByNameFromTeam(team, roles.remove)
    removePrincipalsFromTeam(team, principals.remove)
    removePermissionsFromTeam(team, permissions.remove)
    team.getId
  }

  override def findTeamId(containerId: String, teamName: String): Option[TeamId] = {
    findOptionalXlRoleByName(getContainerCiId(containerId), teamName).map(_.getId)
  }

  override def findTeamById(containerId: TeamId, teamId: TeamId): Option[Team] = {
    val ciId = getContainerCiId(containerId)

    roleRepository.findByCiIdAndId(ciId, teamId) match {
      case Some(xlRole: XlRole) =>
        val team: Team = getTeamAttributes(xlRole)
        Option(team)
      case None => None
    }
  }

  override def findTeamByName(containerId: String, teamName: String): Option[Team] = {
    val ciId = getContainerCiId(containerId)

    findOptionalXlRoleByName(ciId, teamName) match {
      case Some(xlRole: XlRole) =>
        val team: Team = getTeamAttributes(xlRole)
        Option(team)
      case None => None
    }
  }

  override def deleteTeam(teamId: TeamId): Unit = {
    roleRolesRepository.deleteByIdRoleId(teamId)
    rolePrincipalsRepository.deleteByIdRoleId(teamId)
    rolePermissionsRepository.deleteByIdRoleId(teamId)
    roleRepository.deleteById(teamId)
  }

  override def addPrincipalsToTeam(teamId: TeamId, principals: Set[TeamId]): Unit = {
    val team = findRequiredTeam(teamId)
    addPrincipalsToTeam(team, principals)
  }

  override def removePrincipalsFromTeam(teamId: TeamId, principals: Set[TeamId]): Unit = {
    findOptionalTeam(teamId).foreach(removePrincipalsFromTeam(_, principals))
  }

  override def addRolesByNameToTeam(teamId: TeamId, roleNames: Set[String]): Unit = {
    val team = findRequiredTeam(teamId)
    addRolesByNameToTeam(team, roleNames)
  }

  override def removeRolesByNameFromTeam(teamId: TeamId, roleNames: Set[String]): Unit = {
    // It's ok if the team isn't there because the goal is to make the specified team no longer contain the roles so mission accomplished
    findOptionalTeam(teamId).foreach(removeRolesByNameFromTeam(_, roleNames))
  }

  override def addPermissionsToTeam(teamId: TeamId, permissions: Set[String]): Unit = {
    val team = findRequiredTeam(teamId)
    addPermissionsToTeam(team, permissions)
  }

  override def removePermissionsFromTeam(teamId: TeamId, permissions: Set[String]): Unit = {
    findOptionalTeam(teamId).foreach(removePermissionsFromTeam(_, permissions))
  }

  private def getContainerCiId(containerId: String): Int = {
    val securityUid = Option(securedCis.getEffectiveSecuredCi(containerId).getSecurityUid)
    Ids.toDbId(securityUid)
  }

  private def findOptionalXlRoleByName(containerCiId: Int, teamName: String): Option[XlRole] = {
    roleRepository.findByCiIdAndNameIgnoreCase(containerCiId, teamName)
  }

  private def findOptionalTeam(teamId: TeamId): Option[XlRole] = {
    roleRepository.findById(teamId).toScala
  }

  private def findRequiredTeam(teamId: TeamId): XlRole = {
    roleRepository.findById(teamId).orElseThrow(() => new NotFoundException(s"Team [$teamId] not found"))
  }

  private def findGlobalRoleIdsByName(roleNames: Set[String]): Set[XlRole] = {
    roleRepository.findByCiIdAndNameIgnoreCaseIn(-1, roleNames.asJava).asScala.toSet
  }

  private def addPrincipalsToTeam(team: XlRole, principals: Set[String]): Unit = {
    val teamPrincipals = principals.map(new XlRolePrincipal(team, _))
    rolePrincipalsRepository.saveAll(teamPrincipals.asJava)
  }

  private def removePrincipalsFromTeam(team: XlRole, principals: Set[TeamId]): Unit = {
    val ids = principals.map(principal => new XlRolePrincipalId(team.getId, principal)).asJava
    rolePrincipalsRepository.deleteAllById(ids)
  }

  private def addRolesByNameToTeam(team: XlRole, roleNames: Set[String]): Unit = {
    val globalRoleIds: Set[XlRole] = findGlobalRoleIdsByName(roleNames)
    val teamRoles = globalRoleIds.map(new XlRoleRole(team, _))
    roleRolesRepository.saveAll(teamRoles.asJava)
  }

  private def removeRolesByNameFromTeam(team: XlRole, roleNames: Set[String]): Unit = {
    val memberRoles = findGlobalRoleIdsByName(roleNames)
    val ids = memberRoles.map(memberRole => new XlRoleRoleId(team.getId, memberRole.getId)).asJava
    roleRolesRepository.deleteAllById(ids)
  }

  private def addPermissionsToTeam(team: XlRole, permissions: Set[String]): Unit = {
    val teamPermissions = permissions.map(new XlRolePermission(team, _))
    rolePermissionsRepository.saveAll(teamPermissions.asJava)
  }

  private def removePermissionsFromTeam(team: XlRole, permissions: Set[String]): Unit = {
    val ids = permissions.map(permission => new XlRolePermissionId(team, permission)).asJava
    rolePermissionsRepository.deleteAllById(ids)
  }

  private def getTeamAttributes(xlRole: XlRole) = {
    val teamId = xlRole.getId
    val memberRoleNames = roleRolesRepository.findByIdRoleId(teamId).asScala.map(_.getMemberRole.getName).asJava
    val permissions = rolePermissionsRepository.findByIdRoleId(teamId).asScala.map(_.getId.getPermissionName).asJava
    val principals = rolePrincipalsRepository.findByIdRoleId(teamId).asScala.map(_.getId.getPrincipalName).asJava
    val team = new Team()
    team.setId(teamId)
    team.setTeamName(xlRole.getName)
    team.setPermissions(permissions)
    team.setMembers(principals)
    team.setRoles(memberRoleNames)
    team
  }

}

