package com.xebialabs.xlrelease.repository.sql

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, ItemInUseException}
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.PropertyConfiguration.PROPERTY_CARDINALITY
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint
import com.xebialabs.xlrelease.repository.query.ReleaseBasicData
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.ConfigurationRow
import com.xebialabs.xlrelease.repository.sql.persistence.configuration._
import com.xebialabs.xlrelease.repository.sql.persistence.data.ConfigurationReferenceRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence}
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, Ids}
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.repository.JobRepository
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import com.xebialabs.xlrelease.utils.TypeHelper
import com.xebialabs.xlrelease.validation.{ExtendedValidationContextImpl, XlrValidationsFailedException}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.Pageable

import java.util.{Optional, List => JList}
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@IsTransactional
class SqlConfigurationRepository(configurationPersistence: ConfigurationPersistence,
                                 releaseConfigurationReferencePersistence: ReleaseConfigurationReferencePersistence,
                                 triggerConfigurationReferencePersistence: TriggerConfigurationReferencePersistence,
                                 sqlRepositoryAdapter: SqlRepositoryAdapter,
                                 folderPersistence: FolderPersistence,
                                 jobRepository: JobRepository) extends ConfigurationRepository
  with Logging {

  private val RELEASE_REFERENCE_TYPE: String = "Release"
  private val TRIGGER_REFERENCE_TYPE: String = "Trigger"
  private val TASK_REFERENCE_TYPE: String = "Task"
  private val mapper = new ObjectMapper

  @Timed
  override def create[T <: BaseConfiguration](configuration: T): T = {
    create(configuration, getFolderCiUid(configuration.getFolderId))
  }

  @Timed
  override def create[T <: BaseConfiguration](configuration: T, folderCiUid: CiUid): T = {
    checkCardinality(configuration)
    validate(configuration)
    interceptCreate(configuration)
    configurationPersistence.insert(configuration, folderCiUid)
    afterCreate(configuration)
    configuration
  }

  private def getFolderCiUid(folderId: String): CiUid = {
    folderId match {
      case null => null
      case folderId => folderPersistence.getUid(folderId)
    }
  }

  private def checkCardinality[T <: BaseConfiguration](configuration: T): Unit = {
    val configurationType = Type.valueOf(classOf[BaseConfiguration])
    import com.xebialabs.xlrelease.repository.sql.SqlConfigurationRepository._

    def groupType[C <: BaseConfiguration](configuration: C): Option[Type] = {
      val superClasses = configuration.getType.getDescriptor.getSuperClasses
      // the first one where cardinality matches the cardinality of the config itself
      superClasses.asScala.find { superType =>
        superType.isSubTypeOf(configurationType) && superType.hasCardinality &&
          superType.getDescriptor.getPropertyDescriptor(PROPERTY_CARDINALITY).getDefaultValue.asInstanceOf[Int] == configuration.getCardinality
      }
    }

    if (configuration.hasCardinality) {
      val configGroupType = groupType(configuration).getOrElse(configuration.getType)
      val foundConfigs = findAllByTypeAndTitle[BaseConfiguration](configGroupType, title = null, configuration.getFolderId, folderOnly = false).asScala
        .filter(_.getFolderId == configuration.getFolderId)
      val cardinality = configuration.getCardinality
      if (cardinality.isDefined && foundConfigs.size >= cardinality.get) {
        val folderId = if (configuration.getFolderId == null) Ids.ROOT_FOLDER_ID else configuration.getFolderId
        throw new ItemAlreadyExistsException(s"There are already ${foundConfigs.length} items of type '$configGroupType' defined on the folder '$folderId'")
      }
    }
  }

  @Timed
  override def find[T <: BaseConfiguration](configurationId: String): Option[T] =
    configurationPersistence.findById(configurationId)
      .flatMap(readConfiguration[T])

  @Timed
  override def read[T <: BaseConfiguration](configurationId: String): T = {
    this.find(configurationId)
      .getOrElse(throw new NotFoundException(s"Configuration $configurationId not found"))
  }

  @Timed
  def getReleaseConfigurations[T <: BaseConfiguration](releaseUid: CiUid): Seq[T] = {
    releaseConfigurationReferencePersistence.findAllByUid(releaseUid).flatMap(readConfiguration[T])
  }

  @Timed
  override def update[T <: BaseConfiguration](configuration: T): T = {
    update(configuration, getFolderCiUid(configuration.getFolderId))
  }

  @Timed
  override def update[T <: BaseConfiguration](configuration: T, folderCiUid: CiUid): T = {
    validate(configuration)
    interceptUpdate(configuration)
    configurationPersistence.update(configuration)
    afterUpdate(configuration)
    configuration
  }

  @Timed
  override def findByIds[T <: BaseConfiguration](configurationIds: JList[String]): JList[T] =
    configurationPersistence.findByIds(configurationIds.asScala.toSeq)
      .flatMap(readConfiguration[T])
      .toBuffer
      .asJava


  @Timed
  override def findAllByType[T <: BaseConfiguration](ciType: Type): JList[T] =
    configurationPersistence.findByTypes(getAllSubTypesOf(ciType))
      .flatMap(readConfiguration[T])
      .toBuffer
      .asJava

  @Timed
  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String): JList[T] = {
    findAllByTypeAndTitle(ciType, title, folderId = null, folderOnly = false)
  }


  @Timed
  override def findAllByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String = null, folderId: String, folderOnly: Boolean): JList[T] = {
    configurationPersistence.findByTypesTitleAndFolder(
      getAllSubTypesOf(ciType),
      Option(title).filter(_.trim.nonEmpty),
      Option(folderId).map(if (folderOnly) Right.apply else Left.apply)
    ).flatMap(readConfiguration[T]).toBuffer.asJava
  }

  private def readConfiguration[T <: BaseConfiguration](row: ConfigurationRow): Option[T] = {
    row match {
      case (folderIdOpt, rawConfiguration) =>
        sqlRepositoryAdapter.deserialize[T](rawConfiguration, cacheResult = false).map { conf =>
          folderIdOpt.fold(conf) { folderId =>
            conf.setFolderId(folderId.absolute)
            conf
          }
        }
    }
  }

  @Timed
  def findConfigurationTitleById(configId: String): String = {
    configurationPersistence.findConfigurationTitleById(configId)
  }

  @Timed
  @IsReadOnly
  def existsByTypeAndTitle[T <: BaseConfiguration](ciType: Type, title: String = null): Boolean = {
    configurationPersistence.existsByTypeAndTitle(getAllSubTypesOf(ciType), title)
  }

  @Timed
  override def findFirstByType[T <: BaseConfiguration](ciType: Type): Optional[T] = {
    findFirstByType(ciType, ResolutionContext.GLOBAL)
  }

  @Timed
  override def findFirstByType[T <: BaseConfiguration](ciType: Type, context: ResolutionContext): Optional[T] = {
    configurationPersistence.findFirstByTypes(getAllSubTypesOf(ciType), context.folderId)
      .flatMap(readConfiguration[T])
      .asJava
  }

  @Timed
  @IsReadOnly
  override def exists(configurationId: String): Boolean = configurationPersistence.exists(configurationId)

  private def getConfigReferencedRows(configurationId: String, configurationReferencePersistence: ConfigurationReferencePersistence): Seq[ConfigurationReferenceRow] = {
    if (configurationReferencePersistence.isReferenced(configurationId)) {
      configurationReferencePersistence.getReferencingEntities(configurationId)
    } else {
      Seq.empty
    }
  }

  @Timed
  override def delete(configurationId: String): Unit = {
    if (exists(configurationId)) {
      // TODO reference related logic should go to the service where you can call multiple repositories and do the validation
      val config: BaseConfiguration = read(configurationId)
      val allReferences = if (config.getType.isSubTypeOf(Type.valueOf(classOf[JobRunner]))) {
        getJobRunnerReferences(configurationId)
      } else {
        val configReferencesInRelease = getConfigReferencedRows(configurationId, releaseConfigurationReferencePersistence).map(configRef =>
          ReferencedEntity(configRef.id, configRef.title, RELEASE_REFERENCE_TYPE))
        val configReferencesInTrigger = getConfigReferencedRows(configurationId, triggerConfigurationReferencePersistence).map(configRef =>
          ReferencedEntity(configRef.id, configRef.title, TRIGGER_REFERENCE_TYPE))
        configReferencesInRelease ++ configReferencesInTrigger
      }

      allReferences match {
        case Seq() =>
          interceptDelete(configurationId)
          configurationPersistence.delete(configurationId)
          afterDelete(configurationId)
        case _ =>
          mapper.registerModule(DefaultScalaModule)
          val json = mapper.writeValueAsString(allReferences)
          throw new ItemInUseException("%s", json)
      }
    }
  }

  def getJobRunnerReferences(configurationId: String): Seq[ReferencedEntity] = {
    val jobFilters = new JobFilters()
    jobFilters.runnerId = configurationId
    val jobs = jobRepository.findAllJobOverview(jobFilters, Pageable.unpaged())
    jobs.getContent.asScala.map(overview => {
      ReferencedEntity(overview.taskId, overview.taskTitle, TASK_REFERENCE_TYPE)
    }).toSeq
  }

  @Timed
  override def getReferenceReleases(configurationId: String): JList[ReleaseBasicData] = {
    releaseConfigurationReferencePersistence.getReferencingEntities(configurationId).map(_.asReleaseData).asJava
  }

  private def getAllSubTypesOf[T <: BaseConfiguration](ciType: Type) = {
    TypeHelper.getAllSubtypesOf(ciType).map(_.toString)
  }

  @Timed
  override def getAllTypes: Seq[String] = configurationPersistence.findAllConfigurationTypes

  @Timed
  override def deleteByTypes(ciTypes: Seq[String]): Unit = {
    val configurationCiUids = configurationPersistence.findUidsByTypes(ciTypes)
    releaseConfigurationReferencePersistence.deleteRefsByConfigurationUids(configurationCiUids)
    triggerConfigurationReferencePersistence.deleteRefsByConfigurationUids(configurationCiUids)
    configurationPersistence.deleteByTypes(ciTypes)
  }

  @Timed
  override def findAllNonInheritedReleaseReferences(folderId: String, releaseStatus: Seq[ReleaseStatus]): Seq[String] = {
    val parameters: Seq[String] = releaseStatus.map(_.value())
    releaseConfigurationReferencePersistence.findAllNonInheritedReferences(folderId, parameters)
  }

  @Timed
  override def findAllNonInheritedTriggerReferences(folderId: String): Seq[String] = {
    triggerConfigurationReferencePersistence.findAllNonInheritedReferences(folderId, Seq.empty[String])
  }

  private def validate(configuration: BaseConfiguration): Unit = {
    val passVarUsagePoints = ListBuffer.empty[UsagePoint]
    configuration match {
      case conf: Configuration =>
        for {
          usagePoints <- ConfigurationVariableService.getUsagePointsByVars(Seq(conf)).values
          up <- usagePoints
        } {
          val targetProperty = up.getTargetProperty
          val desc = targetProperty.getDescriptor
          checkArgument(desc.getKind == PropertyKind.STRING && desc.isPassword,
            "Non-password-type variables in configuration variableMapping are not supported")
          // Hack warning: temporarily set variableMapped password variables to some value
          // to avoid validation errors on required password properties
          // ideally we should make platform validators aware of variableMapping
          targetProperty.setValue("<temporary_value>")
          passVarUsagePoints += up
        }
      case _ =>
    }

    val desc = configuration.getType.getDescriptor
    val extendedValidationContext = new ExtendedValidationContextImpl(configuration)
    desc.validate(extendedValidationContext, configuration)
    val messages = extendedValidationContext.getMessages
    if (!messages.isEmpty) {
      configuration.get$validationMessages().addAll(messages)
    }
    if (!configuration.get$validationMessages().isEmpty) {
      throw new XlrValidationsFailedException(configuration)
    }

    passVarUsagePoints.toList.foreach { up =>
      up.getTargetProperty.setValue(null)
    }
  }

}

object SqlConfigurationRepository {

  private implicit class TypeOps(xlType: Type) {
    def hasCardinality: Boolean = hasProperty(PROPERTY_CARDINALITY)

    def hasProperty(propertyName: String): Boolean = xlType.getDescriptor.getPropertyDescriptor(propertyName) != null
  }

  private implicit class BaseConfigurationOps(ci: BaseConfiguration) {
    private val ciType = ci.getType

    def hasCardinality: Boolean = ci.hasProperty(PROPERTY_CARDINALITY)

    def getCardinality: Option[Int] = {
      if (hasCardinality) {
        Some(ciType.getDescriptor.getPropertyDescriptor(PROPERTY_CARDINALITY).getDefaultValue.asInstanceOf[Int])
      } else {
        None
      }
    }
  }

}

private[sql] case class ReferencedEntity(id: String, title: String, referenceType: String)
