package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{GateCondition, GateTask}
import com.xebialabs.xlrelease.repository.GateConditionRepository
import com.xebialabs.xlrelease.repository.Ids.{releaseIdFrom, taskIdFrom}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.{DependencyPersistence, ReleasePersistence}
import com.xebialabs.xlrelease.service.TaskConcurrencyService
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired

import java.util.{List => JList}

@IsTransactional
class SqlGateConditionRepository(val releasePersistence: ReleasePersistence,
                                 val dependencyPersistence: DependencyPersistence,
                                 val repositoryAdapter: SqlRepositoryAdapter)
  extends GateConditionRepository with BaseReleaseItemRepository {

  @Autowired
  var taskConcurrencyService: TaskConcurrencyService = _

  @Timed
  override def delete(gateCondition: GateCondition): Unit = {
    val release = getRelease(releaseIdFrom(gateCondition.getId), gateCondition.getId)
    val gateTask = release.getTask(taskIdFrom(gateCondition.getId.normalized)).asInstanceOf[GateTask]

    gateTask.getConditions.remove(gateCondition)
    releasePersistence.update(release)
  }

  @Timed
  override def update(updated: GateCondition, newVariables: JList[Variable]): GateCondition = {
    val release = getRelease(releaseIdFrom(updated.getId), updated.getId)
    release.addVariables(newVariables)
    val gateTask = release.getTask(taskIdFrom(updated.getId.normalized)).asInstanceOf[GateTask]
    gateTask.updateCondition(updated)
    taskConcurrencyService.updateLastModifiedDetails(gateTask)
    releasePersistence.update(release)
    updated
  }

  @Timed
  override def create(gateCondition: GateCondition, newVariables: JList[Variable]): GateCondition = {
    val release = getRelease(releaseIdFrom(gateCondition.getId), gateCondition.getId)
    release.addVariables(newVariables)
    val gateTask = release.getTask(taskIdFrom(gateCondition.getId.normalized)).asInstanceOf[GateTask]
    gateTask.addCondition(gateCondition)
    releasePersistence.update(release)
    gateCondition
  }

  @Timed
  override def findById(gateConditionId: CiId): GateCondition = {
    val release = getRelease(releaseIdFrom(gateConditionId), gateConditionId)
    val gateTask = release.getTask(taskIdFrom(gateConditionId.normalized)).asInstanceOf[GateTask]
    notNull(gateTask.getCondition(gateConditionId.normalized), gateConditionId)
  }
}
