package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Dialect, MSSQLDialect}
import org.springframework.util.StringUtils

case class WhereClause(clause: String = null, params: Map[String, Any] = Map()) {

  import com.xebialabs.xlrelease.repository.sql.query.WhereClause._

  private def combine(op: String, other: WhereClause): WhereClause = {
    (trim(clause), trim(other.clause)) match {
      case (Some(x), Some(y)) => WhereClause(s"($x $op $y)", params ++ other.params)
      case (None, Some(y)) => WhereClause(y, other.params)
      case (Some(x), None) => WhereClause(x, params)
      case (None, None) => WhereClause(null, Map[String, Any]())
    }
  }

  def and(clause: String, params: (String, Any)*): WhereClause = {
    this.and(WhereClause(clause, Map(params: _*)))
  }

  def or(clause: String, params: (String, Any)*): WhereClause = {
    this.or(WhereClause(clause, Map(params: _*)))
  }

  def and(others: WhereClause*): WhereClause = {
    others.foldLeft(this)((prev, other) => prev.combine("AND", other))
  }

  def or(others: WhereClause*) =
    others.foldLeft(this)((prev, other) => prev.combine("OR", other))

  def andLike(column: String, key: String, value: String)(implicit dialect: Dialect): WhereClause = {
    this.and(like(column, key, value))
  }

  def orLike(column: String, key: String, value: String)(implicit dialect: Dialect): WhereClause = {
    this.or(like(column, key, value))
  }

}


object WhereClause {

  def apply(clause: String, params: (String, Any)*): WhereClause = {
    WhereClause(clause, Map(params: _*))
  }

  implicit class WhereClauseForOps(val wc: WhereClause) extends AnyVal {
    def nonEmpty: Boolean = !isEmpty

    def isEmpty: Boolean = {
      trim(wc.clause).isEmpty
    }

    def flatMap[U](f: WhereClause => Option[U]): Option[U] = {
      f(wc)
    }

    def map[U](f: WhereClause => U): Option[U] = {
      Option(f(wc))
    }

    def foreach[U](f: WhereClause => Unit): Unit = {
      f(wc)
    }
  }


  import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect

  private def escapeStatement: String = "escape '\\'"

  private def escapeSquareBrackets(value: String): String = value.replaceAll("\\[", "\\\\[").replaceAll("\\]", "\\\\]")

  def trim(param: String): Option[String] = {
    if (StringUtils.hasText(param)) Some(param.strip()) else None
  }

  def like(column: String, key: String, value: String)(implicit dialect: Dialect): WhereClause = {
    if (value != null && !value.isEmpty) {
      dialect match {
        case MSSQLDialect(_) =>
          WhereClause(
            s"LOWER($column) LIKE :$key $escapeStatement",
            Map(key -> s"%${escapeSquareBrackets(value.toLowerCase())}%")
          )
        case _ =>
          WhereClause(
            s"LOWER($column) LIKE :$key",
            Map(key -> s"%${value.toLowerCase()}%")
          )
      }
    } else {
      WhereClause()
    }
  }

}
