package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.actors.cluster.ClusterMember
import com.xebialabs.xlrelease.db.sql.DatabaseInfo
import com.xebialabs.xlrelease.db.sql.DatabaseInfo.MySql
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.ClusterMembersPersistence._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CLUSTER_MEMBERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.Timestamp
import java.time.Instant

@IsTransactional
class ClusterMembersPersistence(val jdbcTemplate: JdbcTemplate,
                                val dialect: Dialect,
                                dbInfo: DatabaseInfo) extends SqlRepository with PersistenceSupport {

  private def systemColumn: String = dbInfo match {
    case MySql(_) => s"`${CLUSTER_MEMBERS.SYSTEM}`"
    case _ => CLUSTER_MEMBERS.SYSTEM
  }

  private val STMT_CREATE: String =
    s"""|INSERT INTO ${CLUSTER_MEMBERS.TABLE} (
        |   ${CLUSTER_MEMBERS.DATACENTER},
        |   ${CLUSTER_MEMBERS.PROTOCOL},
        |   $systemColumn,
        |   ${CLUSTER_MEMBERS.HOST},
        |   ${CLUSTER_MEMBERS.MANAGEMENT_PORT},
        |   ${CLUSTER_MEMBERS.PORT},
        |   ${CLUSTER_MEMBERS.TTL}
        |) VALUES (
        |   :datacenter,
        |   :protocol,
        |   :system,
        |   :host,
        |   :managementPort,
        |   :port,
        |   :ttl
        |)
     """.stripMargin

  private val STMT_UPDATE_TTL: String =
    s"""|UPDATE ${CLUSTER_MEMBERS.TABLE}
        |SET
        |   ${CLUSTER_MEMBERS.TTL} = :ttl
        |WHERE
        |   ${CLUSTER_MEMBERS.DATACENTER} = :datacenter AND
        |   ${CLUSTER_MEMBERS.PROTOCOL} = :protocol AND
        |   $systemColumn = :system AND
        |   ${CLUSTER_MEMBERS.HOST} = :host AND
        |   ${CLUSTER_MEMBERS.MANAGEMENT_PORT} = :managementPort AND
        |   ${CLUSTER_MEMBERS.PORT} = :port
     """.stripMargin


  private val STMT_DELETE: String =
    s"""|DELETE FROM ${CLUSTER_MEMBERS.TABLE}
        |WHERE
        |   ${CLUSTER_MEMBERS.DATACENTER} = :datacenter AND
        |   ${CLUSTER_MEMBERS.PROTOCOL} = :protocol AND
        |   $systemColumn = :system AND
        |   ${CLUSTER_MEMBERS.HOST} = :host AND
        |   ${CLUSTER_MEMBERS.MANAGEMENT_PORT} = :managementPort AND
        |   ${CLUSTER_MEMBERS.PORT} = :port
     """.stripMargin

  private val STMT_SELECT_ACTIVE: String =
    s"""|SELECT
        |   ${CLUSTER_MEMBERS.DATACENTER},
        |   ${CLUSTER_MEMBERS.PROTOCOL},
        |   $systemColumn,
        |   ${CLUSTER_MEMBERS.HOST},
        |   ${CLUSTER_MEMBERS.MANAGEMENT_PORT},
        |   ${CLUSTER_MEMBERS.PORT}
        |FROM ${CLUSTER_MEMBERS.TABLE}
        |WHERE
        |   ${CLUSTER_MEMBERS.TTL} > :ttl
     """.stripMargin


  private val STMT_EXISTS: String =
    s"""|SELECT 1 FROM ${CLUSTER_MEMBERS.TABLE}
        |WHERE
        |   ${CLUSTER_MEMBERS.DATACENTER} = :datacenter AND
        |   ${CLUSTER_MEMBERS.PROTOCOL} = :protocol AND
        |   $systemColumn = :system AND
        |   ${CLUSTER_MEMBERS.HOST} = :host AND
        |   ${CLUSTER_MEMBERS.MANAGEMENT_PORT} = :managementPort AND
        |   ${CLUSTER_MEMBERS.PORT} = :port
     """.stripMargin

  private val STMT_DELETE_EXPIRED =
    s"""|DELETE FROM ${CLUSTER_MEMBERS.TABLE}
        |WHERE
        |   ${CLUSTER_MEMBERS.TTL} < :ttl
     """.stripMargin

  def create(row: ClusterMemberRow): Unit = {
    val params = row.key.toParamsMap ++ Map("ttl" -> Timestamp.from(row.ttl))
    sqlInsert(STMT_CREATE, params)
  }

  def updateTtl(key: ClusterMemberKey, newTtl: Instant): Boolean = {
    val params = key.toParamsMap ++ Map("ttl" -> Timestamp.from(newTtl))

    sqlUpdate(STMT_UPDATE_TTL, params, _ == 1)
  }

  def delete(key: ClusterMemberKey): Unit = {
    val params = key.toParamsMap
    sqlUpdate(STMT_DELETE, params, _ => ())
  }

  @IsReadOnly
  def listActiveMembers(ttl: Instant): Seq[ClusterMemberKey] = {
    sqlQuery(STMT_SELECT_ACTIVE, params("ttl" -> Timestamp.from(ttl)), clusterMembersMapper).toSeq
  }

  @IsReadOnly
  def exists(key: ClusterMemberKey): Boolean = {
    val params = key.toParamsMap
    sqlQuery(STMT_EXISTS, params, _.getInt(1) > 0).exists(identity)
  }

  def deleteExpiredMembers(expiresAt: Instant): Unit = {
    val params = Map("ttl" -> Timestamp.from(expiresAt))
    sqlUpdate(STMT_DELETE_EXPIRED, params, _ => ())
  }
}


object ClusterMembersPersistence {

  case class ClusterMemberKey(datacenter: String, protocol: String, system: String, host: String, managementPort: Int, port: Int) {

    def toMember: ClusterMember = ClusterMember(datacenter, protocol, system, host, managementPort, port)

    def toParamsMap: Map[String, Any] = Map(
      "datacenter" -> datacenter,
      "protocol" -> protocol,
      "system" -> system,
      "host" -> host,
      "managementPort" -> managementPort,
      "port" -> port
    )
  }

  object ClusterMemberKey {

    def from(member: ClusterMember): ClusterMemberKey = {
      ClusterMemberKey(
        member.datacenter,
        member.protocol,
        member.system,
        member.host,
        member.managementPort,
        member.port
      )
    }
  }

  case class ClusterMemberRow(key: ClusterMemberKey, ttl: Instant) {
  }

  private val clusterMembersMapper: RowMapper[ClusterMemberKey] = (rs, _) => {
    ClusterMemberKey(
      datacenter = rs.getString(CLUSTER_MEMBERS.DATACENTER),
      protocol = rs.getString(CLUSTER_MEMBERS.PROTOCOL),
      system = rs.getString(CLUSTER_MEMBERS.SYSTEM),
      host = rs.getString(CLUSTER_MEMBERS.HOST),
      managementPort = rs.getInt(CLUSTER_MEMBERS.MANAGEMENT_PORT),
      port = rs.getInt(CLUSTER_MEMBERS.PORT)
    )
  }
}
