package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.builder.TaskBuilder.newCustomScript
import com.xebialabs.xlrelease.domain.PythonScript.PYTHON_SCRIPT_PROPERTY
import com.xebialabs.xlrelease.domain._

sealed trait TaskTypeConverter[A <: Task, B <: Task] {
  def convert(task: A, newType: NewType, converterService: TaskTypeConverterService): B
}

object TaskTypeConverterSyntax {
  implicit class TaskTypeConverterOps[A <: Task](task: A) {
    def convertTo[B <: Task](newType: NewType, converterService: TaskTypeConverterService)(implicit converter: TaskTypeConverter[A, B]): B = {
      converter.convert(task, newType, converterService)
    }
  }
}

object TaskTypeConverters {
  implicit def toTaskConverter[A <: Task]: TaskTypeConverter[A, Task] = new TaskTypeConverter[A, Task] {
    def convert(task: A, newType: NewType, converterService: TaskTypeConverterService): Task = {
      val updatedTask: Task = Task.fromType(newType.ciType)
      converterService.forceCopySelectedNonContainmentProperties(task, updatedTask, selectedProperties = Seq("script"))
      converterService.copyProperties(task, updatedTask, excludedProperties = Seq.empty)
      updatedTask
    }
  }

  implicit def toCustomScriptTaskConverter[A <: Task]: TaskTypeConverter[A, CustomScriptTask] = new TaskTypeConverter[A, CustomScriptTask] {
    def convert(task: A, newType: NewType, converterService: TaskTypeConverterService): CustomScriptTask = {
      val updatedTask: CustomScriptTask = newCustomScript(newType.customScriptType.get.toString).withId(task.asInstanceOf[Task].getId).build
      converterService.copyPythonScriptProperties(task, updatedTask)
      converterService.copyProperties(task, updatedTask, excludedProperties = Seq(PYTHON_SCRIPT_PROPERTY))
      updatedTask
    }
  }

  implicit def toContainerTaskConverter[A <: Task]: TaskTypeConverter[A, ContainerTask] =
    new TaskTypeConverter[A, ContainerTask] {
      def convert(task: A, newType: NewType, converterService: TaskTypeConverterService): ContainerTask = {
        val updatedTask: ContainerTask = Task.fromType(newType.containerType.get)
        converterService.copyContainerTaskProperties(task, updatedTask)
        converterService.copyProperties(task, updatedTask, excludedProperties = Seq.empty)
        updatedTask
      }
    }
}


