package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_RUNNER, RUNNER_REGISTRATION, VIEW_RUNNER}
import com.xebialabs.xlrelease.security.sql.SqlRoleService
import com.xebialabs.xlrelease.security.{ReleaseRolePermissionRepository, RolePermission}
import com.xebialabs.xlrelease.service.UserTokenService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.data.domain.PageRequest
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import java.util.HashSet
import scala.jdk.CollectionConverters.CollectionHasAsScala

/**
 * Replace runner#registration permission with runner#edit and runner#view permissions.
 */
@Component
class XLRelease2610RunnerPermissionsUpgrade @Autowired()(val userTokenService: UserTokenService,
                                                         val roleService: SqlRoleService,
                                                         val releaseRolePermissionRepository: ReleaseRolePermissionRepository,
                                                         @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                         @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
   extends Upgrade with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "26.1.0#2")

  override def doUpgrade(): Boolean = {
    val roles = roleService.getRolesForPermission(RUNNER_REGISTRATION.getPermissionName)
    val rolePermissions = releaseRolePermissionRepository.getGlobalRolePermissions(roles.asScala.map(_.getId).toSeq)
    info(s"Found ${rolePermissions.size} roles with 'runner#registration' permission to update.")
    rolePermissions.foreach(rolePermission => {
      if (rolePermission.permissionNames.contains(RUNNER_REGISTRATION.getPermissionName)) {
        val updatedPermissions = new HashSet(rolePermission.permissionNames)
        updatedPermissions.remove(RUNNER_REGISTRATION.getPermissionName)
        updatedPermissions.add(EDIT_RUNNER.getPermissionName)
        updatedPermissions.add(VIEW_RUNNER.getPermissionName)
        val updatedRolePermission = RolePermission(rolePermission.roleId, updatedPermissions)
        releaseRolePermissionRepository.updateGlobalRolePermissions(updatedRolePermission)
      }
    })

    val tokensToUpdate = userTokenService.findByPermission("runner#registration", PageRequest.of(0, 1000))
    info(s"Found ${tokensToUpdate.getTotalElements} user tokens with 'runner#registration' permission to update.")
    tokensToUpdate.getContent.forEach { token =>
      val updatePermissions = token.userTokenPermission.getGlobalPermissions()
      updatePermissions.remove("runner#registration")
      updatePermissions.add("runner#edit")
      updatePermissions.add("runner#view")
      userTokenService.updateTokenPermissions(token.getCiUid, updatePermissions.asScala.toSet)
    }
    true
  }
}
