import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { XlrTags } from './react-tags/xlr-tags.component';
import { DEFAULT_CLASSNAMES } from './react-tags/xlr-tags-constants';

import './xlr-members-tags.less';

/**
 * @description
 * React component to manage tag list input with auto-completion for members.
 *
 * @example
 * <XlrMembersTags
 *    classNames={{tagInput: 'tag-new principal', selected: 'tag principal'}}
 *    placeholder={'Add principal...'}
 *    tags={tags}
 *    suggestions={suggestions}
 *    handleDelete={this.onMemberDelete}
 *    handleAddition={this.onMemberChange}
 *    readOnly={false}
 * />
 *
 * handleDelete(i) {
 *    const { tags } = this.state;
 *    this.setState({
 *      tags: tags.filter((tag, index) => index !== i),
 *    });
 * }
 *
 * handleAddition(tag) {
 *    this.setState(state => ({ tags: [...state.tags, tag] }));
 * }
 *
 */
export class XlrMembersTags extends Component {
    static propTypes = {
        allowAddOnlyFromSuggestion: PropTypes.bool,
        classNames: PropTypes.object,
        handleAddition: PropTypes.func.isRequired,
        handleDelete: PropTypes.func.isRequired,
        id: PropTypes.string,
        label: PropTypes.string,
        maxLength: PropTypes.number,
        name: PropTypes.string,
        placeholder: PropTypes.string,
        readOnly: PropTypes.bool,
        suggestions: PropTypes.arrayOf(PropTypes.object),
        tags: PropTypes.arrayOf(PropTypes.object),
    };

    static defaultProps = {
        allowAddOnlyFromSuggestion: false,
        classNames: { ...DEFAULT_CLASSNAMES, tagInput: 'tag-new principal', selected: 'tag principal' },
        label: 'name',
        placeholder: 'Add...',
        suggestions: [],
        readOnly: false,
        tags: [],
    };

    renderSuggestion = (item) => {
        const { isVirtual } = item;
        return (
            <div className={'suggestion-wrapper'}>
                {!isVirtual && <span className={'xl-icon tag-icon'} />}
                <span className={'tag-label'}>{item[this.props.label]}</span>
            </div>
        );
    };

    render() {
        const { allowAddOnlyFromSuggestion, handleAddition, handleDelete, id, label, maxLength, name, placeholder, readOnly, suggestions, tags } = this.props;
        const classNames = { ...XlrMembersTags.defaultProps.classNames, ...this.props.classNames };

        return (
            <React.Fragment>
                <XlrTags
                    allowAddFromPaste={false}
                    allowAddOnlyFromSuggestion={allowAddOnlyFromSuggestion}
                    autofocus={false}
                    classNames={classNames}
                    handleAddition={handleAddition}
                    handleDelete={handleDelete}
                    id={id}
                    labelField={label}
                    maxLength={maxLength}
                    minQueryLength={1}
                    name={name}
                    placeholder={placeholder}
                    readOnly={readOnly}
                    renderSuggestion={this.renderSuggestion}
                    suggestions={suggestions}
                    tags={tags}
                />
            </React.Fragment>
        );
    }
}
