import React from 'react';
import angular from 'angular';
import { DotButton, DotEmptyFilterState, DotEmptyState } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import * as angularAccessor from '@xlr-ui/app/features/common/services/angular-accessor';
import { UploadService } from '@xlr-ui/app/features/tasks/types/angular';
import { lazyInjector } from '@xlr-ui/app/components/angular2react/lazy-injector';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { TemplateListEmptyStateContent, TemplateListEmptyStateProps } from './template-list-empty-state.component';

const { navigateTo } = navigation.actions;

describe('TemplateListEmptyState', () => {
    let wrapper: ReactWrapper;

    const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance<UploadService, [name: unknown]>;
    const uploadService = {
        reset: jest.fn(),
    };

    const defaultProps: TemplateListEmptyStateProps = {
        filtered: false,
        hasCreateTemplatePermission: true,
        onImportTemplateSuccess: jest.fn(),
    };

    const dispatch = jest.fn();

    const mount = (props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<TemplateListEmptyStateContent {...props} />, dispatch);
    };

    const getDotEmptyState = () => wrapper.find(DotEmptyState);
    const getDotEmptyFilterState = () => wrapper.find(DotEmptyFilterState);
    const getImportTemplateButton = () => wrapper.find('.empty-state-buttons').find(DotButton).at(0);
    const getNewTemplateButton = () => wrapper.find('.empty-state-buttons').find(DotButton).at(1);

    beforeEach(() => {
        getAngularServiceSpy.mockReturnValue(uploadService as never);

        angular.mock.inject(($injector: never) => {
            lazyInjector.$injector = $injector;
        });
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render without errors', () => {
        mount();
        expect(wrapper.exists()).toBe(true);
    });

    it('should render DotEmptyFilterState', () => {
        mount({ ...defaultProps, filtered: true });
        expect(getDotEmptyFilterState()).toExist();
        expect(getImportTemplateButton()).not.toExist();
        expect(getNewTemplateButton()).not.toExist();
    });

    it('should render DotEmptyState and buttons', () => {
        mount();
        const dotEmptyStateProps = getDotEmptyState().props();
        expect(getDotEmptyFilterState()).not.toExist();
        expect(getDotEmptyState()).toExist();
        expect(dotEmptyStateProps.illustrationId).toBe('add-new-grid');
        expect(dotEmptyStateProps.subtitle).toBe('Design a release process template by creating a new one or importing an existing template.');
        expect(dotEmptyStateProps.title).toBe('Release templates');
        expect(getImportTemplateButton().props().type).toBe('text');
        expect(getImportTemplateButton()).toHaveText('Import template');
        expect(getNewTemplateButton()).toHaveText('New template');
    });

    it('should render disabled buttons when no permission', () => {
        mount({ ...defaultProps, hasCreateTemplatePermission: false });
        const importTemplateButtonProps = getImportTemplateButton().props();
        expect(importTemplateButtonProps.tooltip).toBe('You do not have permission to import a template.');
        expect(importTemplateButtonProps.disabled).toBe(true);
        const newTemplateButtonProps = getNewTemplateButton().props();
        expect(newTemplateButtonProps.tooltip).toBe('You do not have permission to create a new template.');
        expect(newTemplateButtonProps.disabled).toBe(true);
    });

    it('should trigger callback on "new template" when no folder', () => {
        mount();
        getNewTemplateButton().invoke('onClick')?.('' as never);
        expect(dispatch).toHaveBeenCalledWith(navigateTo({ pathSuffix: 'templates/create' }));
    });

    it('should trigger callback on "new template"  when there is folder', () => {
        mount({ ...defaultProps, folder: { id: 'folder-123' } });
        getNewTemplateButton().invoke('onClick')?.('' as never);
        expect(dispatch).toHaveBeenCalledWith(navigateTo({ pathSuffix: 'folders/folder-123/templates?action=create' }));
    });

    it('should trigger callback for import template button', () => {
        mount();
        getImportTemplateButton().invoke('onClick')?.('' as never);
        expect(uploadService.reset).toHaveBeenCalled();
    });
});
