import React from 'react';
import { DotButton, DotEmptyFilterState, DotEmptyState } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { WorkflowTemplateListEmptyStateContent, WorkflowTemplateListEmptyStateProps } from './workflow-template-list-empty-state.component';

const { navigateTo } = navigation.actions;

describe('WorkflowTemplateListEmptyState', () => {
    let wrapper: ReactWrapper;

    const defaultProps: WorkflowTemplateListEmptyStateProps = {
        filtered: false,
        hasCreateTemplatePermission: true,
    };

    const dispatch = jest.fn();

    const mount = (props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<WorkflowTemplateListEmptyStateContent {...props} />, dispatch);
    };

    const getDotEmptyState = () => wrapper.find(DotEmptyState);
    const getDotEmptyFilterState = () => wrapper.find(DotEmptyFilterState);
    const getNewWorkflowTemplateButton = () => wrapper.find(DotButton);

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render without errors', () => {
        mount();
        expect(wrapper.exists()).toBe(true);
    });

    it('should render DotEmptyFilterState', () => {
        mount({ ...defaultProps, filtered: true });
        expect(getDotEmptyFilterState()).toExist();
        expect(getNewWorkflowTemplateButton()).not.toExist();
    });

    it('should render DotEmptyState and buttons', () => {
        mount();
        const dotEmptyStateProps = getDotEmptyState().props();
        expect(getDotEmptyFilterState()).not.toExist();
        expect(getDotEmptyState()).toExist();
        expect(dotEmptyStateProps.illustrationId).toBe('add-new-grid');
        expect(dotEmptyStateProps.subtitle).toBe('Create lightweight, step-by-step workflows to automate tasks and operations.');
        expect(dotEmptyStateProps.title).toBe('Workflow templates');
        expect(getNewWorkflowTemplateButton()).toHaveText('New workflow template');
    });

    it('should trigger callback when no folder', () => {
        mount();
        getNewWorkflowTemplateButton().invoke('onClick')?.('' as never);
        expect(dispatch).toHaveBeenCalledWith(navigateTo({ pathSuffix: 'workflow-templates/create' }));
    });

    it('should trigger callback when there is folder', () => {
        mount({ ...defaultProps, folder: { id: 'folder-123' } });
        getNewWorkflowTemplateButton().invoke('onClick')?.('' as never);
        expect(dispatch).toHaveBeenCalledWith(navigateTo({ pathSuffix: 'folders/folder-123/workflows/templates?action=create' }));
    });

    it('should render disabled button when no permission', () => {
        mount({ ...defaultProps, hasCreateTemplatePermission: false });
        const newTemplateButtonProps = getNewWorkflowTemplateButton().props();
        expect(newTemplateButtonProps.tooltip).toBe('You do not have permission to create a new workflow template.');
        expect(newTemplateButtonProps.disabled).toBe(true);
    });
});
