"""
Helper functions for building performance result parsers.

Warning: The API of this module is not fixed yet. As a result we will not guarantee backwards compatibility if you use it.
"""

import hashlib

from com.xebialabs.xlt.plugin.api.resultparser import UnexpectedFormatException
from com.xebialabs.xlt.plugin.api.resultparser import ImportFailedException
from com.xebialabs.xlt.plugin.api.testrun import Event

def throw_if_some_failed(input_files, accepted_files):
    """ Determine if some files in the list input_files are not in the accepted_files set. If some files were not accepted then an UnexpectedFormatException is thrown.
    :param input_files: a list of files.
    :param accepted_files: a set of files that was accepted.
    """
    diff = set(input_files).difference(accepted_files)
    if diff:
        msg = "Canceled the import. The following files were not accepted by the test tool: " + ", ".join(map(str, diff))
        raise UnexpectedFormatException(msg, diff)

def sha256(file):
    """ Calculate the SHA-256 checksum of `file`.

    :param file: the file to calculate the checksum for.
    :return: the SHA-256 checksum.
    """
    BUF_SIZE = 65536  # lets read stuff in 64kb chunks!

    with open(file, 'rb') as f:
        sha = hashlib.sha256()
        while True:
            data = f.read(BUF_SIZE)
            if not data:
                break
            sha.update(data)
    return sha.hexdigest()


def sha256_of_files(files):
    """
    Calculate a combined SHA-256 checksum of a set of files.
    :param files:the files to calculate the checksums for.
    :return: a combined checksum for `files`.
    """
    checksums = []
    for file in files:
        checksums.append(sha256(file.getPath()))
    checksums = sorted(checksums)
    sha = hashlib.sha256()
    for checksum in checksums:
        sha.update(checksum)
    return sha.hexdigest()


def performance_test_result_parser(files, parser, test_run_historian):
    """ Parse files with performance results into events.

    :param files: the files containing performance test results.
    :param parser: the test tool specific parser.
    :param test_run_historian: test run historian service that can be used to detect already imported results.
    :return:
    """
    events = []
    originalList = list(files)
    filtered = parser.filter_readable(originalList)
    print "After filtering, %d files are going to be processed" % len(filtered)

    throw_if_some_failed(files, filtered)

    run_key = sha256_of_files(filtered)

    if test_run_historian.isKnownKey(run_key):
        return []

    for file in filtered:
        try:
            performanceResultsSummary = parser.construct_performance_results_summary(file)
        except Exception as e:
            print "Executing python failed:", e
            raise ImportFailedException(e)

        if not performanceResultsSummary:
            # For some tools it is required to skip certain files, such as gatling
            continue

        # first create performance event
        performanceResultEvent = performanceResultsSummary.createPerformanceResultEvent()

        # create import started event
        importStartedEvent = Event.createImportStartedEvent() \
            .setRunKey(run_key)
        parser.enrich_import_started_event(importStartedEvent)

        if importStartedEvent.getTestedAt() == -1:
            importStartedEvent.setTestedAt(file.lastModified())

        importFinishedEvent = Event.createImportFinishedEvent(performanceResultsSummary.getDuration())

        events.append([importStartedEvent, performanceResultEvent, importFinishedEvent])

    return events
