package com.farimarwat.composenativeadmob.nativead

import android.content.Context
import android.graphics.drawable.Drawable
import android.util.Log
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import com.google.android.gms.ads.AdListener
import com.google.android.gms.ads.AdLoader
import com.google.android.gms.ads.AdRequest
import com.google.android.gms.ads.LoadAdError
import com.google.android.gms.ads.nativead.NativeAd
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.withContext

@Composable
fun rememberNativeAdState(
    context: Context,
    adUnitId: String,
    refreshInterval: Long = 60000L,
    onAdLoaded:()->Unit={},
    onAdFailedToLoad:(LoadAdError)->Unit={}
): NativeAd? {
    var state by remember { mutableStateOf<NativeAd?>(null) }
    val adLoader = remember(context, adUnitId) {
        AdLoader.Builder(context, adUnitId)
            .forNativeAd { ad ->
                state = ad
            }
            .withAdListener(object : AdListener() {
                override fun onAdLoaded() {
                    super.onAdLoaded()
                    onAdLoaded()
                }

                override fun onAdFailedToLoad(error: LoadAdError) {
                    super.onAdFailedToLoad(error)
                    onAdFailedToLoad(error)
                }
            })
            .build()
    }

    val adRequest = remember { AdRequest.Builder().build() }

    LaunchedEffect(adLoader) {
        if (state == null) {
            withContext(Dispatchers.IO) {
                adLoader.loadAd(adRequest)
            }
        }
    }

    // Refresh logic
    LaunchedEffect(adLoader, refreshInterval) {
        if (refreshInterval > 0) {
            while (true) {
                delay(refreshInterval)
                state = null
                withContext(Dispatchers.IO) {
                    adLoader.loadAd(adRequest)
                }
            }
        }
    }

    // Cleanup when the composable is disposed
    DisposableEffect(adLoader) {
        onDispose {
            state?.destroy()
        }
    }

    return state
}