/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.CallOptions;
import io.grpc.ClientCall;
import io.grpc.Codec;
import io.grpc.Compressor;
import io.grpc.CompressorRegistry;
import io.grpc.Context;
import io.grpc.Contexts;
import io.grpc.Deadline;
import io.grpc.Decompressor;
import io.grpc.DecompressorRegistry;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.internal.ClientStream;
import io.grpc.internal.ClientStreamListener;
import io.grpc.internal.ClientTransport;
import io.grpc.internal.ContextRunnable;
import io.grpc.internal.FailingClientStream;
import io.grpc.internal.GrpcUtil;
import io.grpc.internal.LogExceptionRunnable;
import io.grpc.internal.NoopClientStream;
import io.grpc.internal.SerializeReentrantCallsDirectExecutor;
import io.grpc.internal.SerializingExecutor;
import java.io.InputStream;
import java.util.concurrent.CancellationException;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

final class ClientCallImpl<ReqT, RespT>
extends ClientCall<ReqT, RespT>
implements Context.CancellationListener {
    private static final Logger log = Logger.getLogger(ClientCallImpl.class.getName());
    private final MethodDescriptor<ReqT, RespT> method;
    private final Executor callExecutor;
    private final Context context;
    private volatile ScheduledFuture<?> deadlineCancellationFuture;
    private final boolean unaryRequest;
    private final CallOptions callOptions;
    private ClientStream stream;
    private volatile boolean cancelListenersShouldBeRemoved;
    private boolean cancelCalled;
    private boolean halfCloseCalled;
    private final ClientTransportProvider clientTransportProvider;
    private ScheduledExecutorService deadlineCancellationExecutor;
    private DecompressorRegistry decompressorRegistry = DecompressorRegistry.getDefaultInstance();
    private CompressorRegistry compressorRegistry = CompressorRegistry.getDefaultInstance();

    ClientCallImpl(MethodDescriptor<ReqT, RespT> method, Executor executor, CallOptions callOptions, ClientTransportProvider clientTransportProvider, ScheduledExecutorService deadlineCancellationExecutor) {
        this.method = method;
        this.callExecutor = executor == MoreExecutors.directExecutor() ? new SerializeReentrantCallsDirectExecutor() : new SerializingExecutor(executor);
        this.context = Context.current();
        this.unaryRequest = method.getType() == MethodDescriptor.MethodType.UNARY || method.getType() == MethodDescriptor.MethodType.SERVER_STREAMING;
        this.callOptions = callOptions;
        this.clientTransportProvider = clientTransportProvider;
        this.deadlineCancellationExecutor = deadlineCancellationExecutor;
    }

    public void cancelled(Context context) {
        this.stream.cancel(Contexts.statusFromCancelled(context));
    }

    ClientCallImpl<ReqT, RespT> setDecompressorRegistry(DecompressorRegistry decompressorRegistry) {
        this.decompressorRegistry = decompressorRegistry;
        return this;
    }

    ClientCallImpl<ReqT, RespT> setCompressorRegistry(CompressorRegistry compressorRegistry) {
        this.compressorRegistry = compressorRegistry;
        return this;
    }

    @VisibleForTesting
    static void prepareHeaders(Metadata headers, DecompressorRegistry decompressorRegistry, Compressor compressor) {
        headers.removeAll(GrpcUtil.MESSAGE_ENCODING_KEY);
        if (compressor != Codec.Identity.NONE) {
            headers.put(GrpcUtil.MESSAGE_ENCODING_KEY, compressor.getMessageEncoding());
        }
        headers.removeAll(GrpcUtil.MESSAGE_ACCEPT_ENCODING_KEY);
        String advertisedEncodings = decompressorRegistry.getRawAdvertisedMessageEncodings();
        if (!advertisedEncodings.isEmpty()) {
            headers.put(GrpcUtil.MESSAGE_ACCEPT_ENCODING_KEY, advertisedEncodings);
        }
    }

    @Override
    public void start(final ClientCall.Listener<RespT> observer, Metadata headers) {
        boolean deadlineExceeded;
        Preconditions.checkState((this.stream == null ? 1 : 0) != 0, (Object)"Already started");
        Preconditions.checkNotNull(observer, (Object)"observer");
        Preconditions.checkNotNull((Object)headers, (Object)"headers");
        if (this.context.isCancelled()) {
            this.stream = NoopClientStream.INSTANCE;
            class ClosedByContext
            extends ContextRunnable {
                ClosedByContext() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public void runInContext() {
                    observer.onClose(Contexts.statusFromCancelled(ClientCallImpl.this.context), new Metadata());
                }
            }
            this.callExecutor.execute(new ClosedByContext());
            return;
        }
        final String compressorName = this.callOptions.getCompressor();
        Compressor compressor = null;
        if (compressorName != null) {
            compressor = this.compressorRegistry.lookupCompressor(compressorName);
            if (compressor == null) {
                this.stream = NoopClientStream.INSTANCE;
                class ClosedByNotFoundCompressor
                extends ContextRunnable {
                    ClosedByNotFoundCompressor() {
                        super(ClientCallImpl.this.context);
                    }

                    @Override
                    public void runInContext() {
                        observer.onClose(Status.INTERNAL.withDescription(String.format("Unable to find compressor by name %s", compressorName)), new Metadata());
                    }
                }
                this.callExecutor.execute(new ClosedByNotFoundCompressor());
                return;
            }
        } else {
            compressor = Codec.Identity.NONE;
        }
        ClientCallImpl.prepareHeaders(headers, this.decompressorRegistry, compressor);
        Deadline effectiveDeadline = this.effectiveDeadline();
        boolean bl = deadlineExceeded = effectiveDeadline != null && effectiveDeadline.isExpired();
        if (!deadlineExceeded) {
            ClientCallImpl.updateTimeoutHeaders(effectiveDeadline, this.callOptions.getDeadline(), this.context.getDeadline(), headers);
            ClientTransport transport = this.clientTransportProvider.get(this.callOptions);
            this.stream = transport.newStream(this.method, headers, this.callOptions);
        } else {
            this.stream = new FailingClientStream(Status.DEADLINE_EXCEEDED);
        }
        if (this.callOptions.getAuthority() != null) {
            this.stream.setAuthority(this.callOptions.getAuthority());
        }
        this.stream.setCompressor(compressor);
        this.stream.start(new ClientStreamListenerImpl(observer));
        this.context.addListener((Context.CancellationListener)this, MoreExecutors.directExecutor());
        if (effectiveDeadline != null && this.context.getDeadline() != effectiveDeadline) {
            this.deadlineCancellationFuture = this.startDeadlineTimer(effectiveDeadline);
        }
        if (this.cancelListenersShouldBeRemoved) {
            this.removeContextListenerAndCancelDeadlineFuture();
        }
    }

    private static void updateTimeoutHeaders(@Nullable Deadline effectiveDeadline, @Nullable Deadline callDeadline, @Nullable Deadline outerCallDeadline, Metadata headers) {
        headers.removeAll(GrpcUtil.TIMEOUT_KEY);
        if (effectiveDeadline == null) {
            return;
        }
        long effectiveTimeout = Math.max(0L, effectiveDeadline.timeRemaining(TimeUnit.NANOSECONDS));
        headers.put(GrpcUtil.TIMEOUT_KEY, effectiveTimeout);
        ClientCallImpl.logIfContextNarrowedTimeout(effectiveTimeout, effectiveDeadline, outerCallDeadline, callDeadline);
    }

    private static void logIfContextNarrowedTimeout(long effectiveTimeout, Deadline effectiveDeadline, @Nullable Deadline outerCallDeadline, @Nullable Deadline callDeadline) {
        if (!log.isLoggable(Level.INFO) || outerCallDeadline != effectiveDeadline) {
            return;
        }
        StringBuilder builder = new StringBuilder();
        builder.append(String.format("Call timeout set to '%d' ns, due to context deadline.", effectiveTimeout));
        if (callDeadline == null) {
            builder.append(" Explicit call timeout was not set.");
        } else {
            long callTimeout = callDeadline.timeRemaining(TimeUnit.NANOSECONDS);
            builder.append(String.format(" Explicit call timeout was '%d' ns.", callTimeout));
        }
        log.info(builder.toString());
    }

    private void removeContextListenerAndCancelDeadlineFuture() {
        this.context.removeListener((Context.CancellationListener)this);
        ScheduledFuture<?> f = this.deadlineCancellationFuture;
        if (f != null) {
            f.cancel(false);
        }
    }

    private ScheduledFuture<?> startDeadlineTimer(Deadline deadline) {
        return this.deadlineCancellationExecutor.schedule(new LogExceptionRunnable(new DeadlineTimer()), deadline.timeRemaining(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
    }

    @Nullable
    private Deadline effectiveDeadline() {
        return ClientCallImpl.min(this.callOptions.getDeadline(), this.context.getDeadline());
    }

    @Nullable
    private static Deadline min(@Nullable Deadline deadline0, @Nullable Deadline deadline1) {
        if (deadline0 == null) {
            return deadline1;
        }
        if (deadline1 == null) {
            return deadline0;
        }
        return deadline0.minimum(deadline1);
    }

    @Override
    public void request(int numMessages) {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        Preconditions.checkArgument((numMessages >= 0 ? 1 : 0) != 0, (Object)"Number requested must be non-negative");
        this.stream.request(numMessages);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void cancel(@Nullable String message, @Nullable Throwable cause) {
        if (message == null && cause == null) {
            cause = new CancellationException("Cancelled without a message or cause");
            log.log(Level.WARNING, "Cancelling without a message or cause is suboptimal", cause);
        }
        if (this.cancelCalled) {
            return;
        }
        this.cancelCalled = true;
        try {
            if (this.stream != null) {
                Status status = Status.CANCELLED;
                if (message != null) {
                    status = status.withDescription(message);
                }
                if (cause != null) {
                    status = status.withCause(cause);
                }
                this.stream.cancel(status);
            }
        }
        finally {
            this.removeContextListenerAndCancelDeadlineFuture();
        }
    }

    @Override
    public void halfClose() {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        Preconditions.checkState((!this.cancelCalled ? 1 : 0) != 0, (Object)"call was cancelled");
        Preconditions.checkState((!this.halfCloseCalled ? 1 : 0) != 0, (Object)"call already half-closed");
        this.halfCloseCalled = true;
        this.stream.halfClose();
    }

    @Override
    public void sendMessage(ReqT message) {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        Preconditions.checkState((!this.cancelCalled ? 1 : 0) != 0, (Object)"call was cancelled");
        Preconditions.checkState((!this.halfCloseCalled ? 1 : 0) != 0, (Object)"call was half-closed");
        try {
            InputStream messageIs = this.method.streamRequest(message);
            this.stream.writeMessage(messageIs);
        }
        catch (Throwable e) {
            this.stream.cancel(Status.CANCELLED.withCause(e).withDescription("Failed to stream message"));
            return;
        }
        if (!this.unaryRequest) {
            this.stream.flush();
        }
    }

    @Override
    public void setMessageCompression(boolean enabled) {
        Preconditions.checkState((this.stream != null ? 1 : 0) != 0, (Object)"Not started");
        this.stream.setMessageCompression(enabled);
    }

    @Override
    public boolean isReady() {
        return this.stream.isReady();
    }

    private class ClientStreamListenerImpl
    implements ClientStreamListener {
        private final ClientCall.Listener<RespT> observer;
        private boolean closed;

        public ClientStreamListenerImpl(ClientCall.Listener<RespT> observer) {
            this.observer = (ClientCall.Listener)Preconditions.checkNotNull(observer, (Object)"observer");
        }

        @Override
        public void headersRead(final Metadata headers) {
            Decompressor decompressor = Codec.Identity.NONE;
            if (headers.containsKey(GrpcUtil.MESSAGE_ENCODING_KEY)) {
                String encoding = headers.get(GrpcUtil.MESSAGE_ENCODING_KEY);
                decompressor = ClientCallImpl.this.decompressorRegistry.lookupDecompressor(encoding);
                if (decompressor == null) {
                    ClientCallImpl.this.stream.cancel(Status.INTERNAL.withDescription(String.format("Can't find decompressor for %s", encoding)));
                    return;
                }
            }
            ClientCallImpl.this.stream.setDecompressor(decompressor);
            class HeadersRead
            extends ContextRunnable {
                HeadersRead() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public final void runInContext() {
                    try {
                        if (ClientStreamListenerImpl.this.closed) {
                            return;
                        }
                        ClientStreamListenerImpl.this.observer.onHeaders(headers);
                    }
                    catch (Throwable t) {
                        Status status = Status.CANCELLED.withCause(t).withDescription("Failed to read headers");
                        ClientCallImpl.this.stream.cancel(status);
                        ClientStreamListenerImpl.this.close(status, new Metadata());
                    }
                }
            }
            ClientCallImpl.this.callExecutor.execute(new HeadersRead());
        }

        @Override
        public void messageRead(final InputStream message) {
            class MessageRead
            extends ContextRunnable {
                MessageRead() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public final void runInContext() {
                    try {
                        if (ClientStreamListenerImpl.this.closed) {
                            return;
                        }
                        try {
                            ClientStreamListenerImpl.this.observer.onMessage(ClientCallImpl.this.method.parseResponse(message));
                        }
                        finally {
                            message.close();
                        }
                    }
                    catch (Throwable t) {
                        Status status = Status.CANCELLED.withCause(t).withDescription("Failed to read message.");
                        ClientCallImpl.this.stream.cancel(status);
                        ClientStreamListenerImpl.this.close(status, new Metadata());
                    }
                }
            }
            ClientCallImpl.this.callExecutor.execute(new MessageRead());
        }

        private void close(Status status, Metadata trailers) {
            this.closed = true;
            ClientCallImpl.this.cancelListenersShouldBeRemoved = true;
            try {
                this.observer.onClose(status, trailers);
            }
            finally {
                ClientCallImpl.this.removeContextListenerAndCancelDeadlineFuture();
            }
        }

        @Override
        public void closed(Status status, Metadata trailers) {
            Deadline deadline = ClientCallImpl.this.effectiveDeadline();
            if (status.getCode() == Status.Code.CANCELLED && deadline != null && deadline.isExpired()) {
                status = Status.DEADLINE_EXCEEDED;
                trailers = new Metadata();
            }
            final Status savedStatus = status;
            final Metadata savedTrailers = trailers;
            class StreamClosed
            extends ContextRunnable {
                StreamClosed() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public final void runInContext() {
                    if (ClientStreamListenerImpl.this.closed) {
                        return;
                    }
                    ClientStreamListenerImpl.this.close(savedStatus, savedTrailers);
                }
            }
            ClientCallImpl.this.callExecutor.execute(new StreamClosed());
        }

        @Override
        public void onReady() {
            class StreamOnReady
            extends ContextRunnable {
                StreamOnReady() {
                    super(ClientCallImpl.this.context);
                }

                @Override
                public final void runInContext() {
                    try {
                        ClientStreamListenerImpl.this.observer.onReady();
                    }
                    catch (Throwable t) {
                        Status status = Status.CANCELLED.withCause(t).withDescription("Failed to call onReady.");
                        ClientCallImpl.this.stream.cancel(status);
                        ClientStreamListenerImpl.this.close(status, new Metadata());
                    }
                }
            }
            ClientCallImpl.this.callExecutor.execute(new StreamOnReady());
        }
    }

    private class DeadlineTimer
    implements Runnable {
        private DeadlineTimer() {
        }

        @Override
        public void run() {
            ClientCallImpl.this.stream.cancel(Status.DEADLINE_EXCEEDED);
        }
    }

    static interface ClientTransportProvider {
        public ClientTransport get(CallOptions var1);
    }
}

