/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security.impl;

import java.security.InvalidAlgorithmParameterException;
import java.security.SecureRandom;
import java.util.Random;
import java.util.random.RandomGenerator;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.security.IdentifierGenerationStrategy;
import net.shibboleth.shared.security.RandomIdentifierParameterSpec;

import org.apache.commons.codec.BinaryEncoder;
import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.binary.StringUtils;

/**
 * Generates a random number of bytes via a {@link Random} source and encodes them into a string using a
 * {@link BinaryEncoder} ({@link Hex} by default).
 */
@ThreadSafe
public class RandomIdentifierGenerationStrategy implements IdentifierGenerationStrategy {

    /** Random number generator. */
    @Nonnull private final RandomGenerator random;

    /** Number of random bytes in the identifier. */
    private final int sizeOfIdentifier;

    /** Encoder used to convert the random bytes in to a string. */
    @Nonnull private final BinaryEncoder encoder;
    
    /**
     * Default constructor.
     * 
     * <p>Initializes the random number source to a new {@link SecureRandom}, size of identifier is set to 16
     * bytes, and the encoder is set to a {@link Hex}.</p>
     */
    public RandomIdentifierGenerationStrategy() {
        random = new SecureRandom();
        sizeOfIdentifier = 16;
        encoder = new Hex();
    }

    /**
     * Constructor.
     * 
     * @param params parameter object, must be a {@link RandomIdentifierParameterSpec}
     * 
     * @throws InvalidAlgorithmParameterException if the parameters are invalid
     */
    public RandomIdentifierGenerationStrategy(@Nonnull final ParameterSpec params)
            throws InvalidAlgorithmParameterException {
        if (params instanceof RandomIdentifierParameterSpec spec) {
            final RandomGenerator rng = spec.source();
            if (rng != null) {
                random = rng;
            } else {
                random = new SecureRandom();
            }

            final Integer size = spec.identifierSize();
            if (size != null) {
                if (size <= 0) {
                    throw new InvalidAlgorithmParameterException("Identifier length must be greater than 0");
                }
                sizeOfIdentifier = size;
            } else {
                sizeOfIdentifier = 16;
            }
            
            final BinaryEncoder enc = spec.identifierEncoder();
            if (enc != null) {
                encoder = enc;
            } else {
                encoder = new Hex();
            }
        } else {
            throw new InvalidAlgorithmParameterException("Invalid or unrecognized RandomParameterSpec");
        }
    }
    
    /** {@inheritDoc} */
    @Nonnull @NotEmpty public String generateIdentifier() {
        return generateIdentifier(true);
    }

    /** {@inheritDoc} */
    @Nonnull @NotEmpty public String generateIdentifier(final boolean xmlSafe) {
        final byte[] buf = new byte[sizeOfIdentifier];
        random.nextBytes(buf);
        try {
            if (xmlSafe) {
                return "_" + StringUtils.newStringUsAscii(encoder.encode(buf));
            }
            return StringUtils.newStringUsAscii(encoder.encode(buf));
        } catch (final EncoderException e) {
            throw new RuntimeException(e);
        }
    }

}