/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security;

import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.security.impl.RandomIdentifierGenerationStrategy;
import net.shibboleth.shared.security.impl.SecureRandomIdentifierGenerationStrategy;
import net.shibboleth.shared.security.impl.Type4UUIDIdentifierGenerationStrategy;

/**
 * Interface for identifier generators. This identifier can be used for things like digital signature identifiers,
 * opaque principal identifiers, etc.
 */
@ThreadSafe
public interface IdentifierGenerationStrategy {

    /**
     * Generates an identifier guaranteed to be XML ID safe.
     * 
     * @return the identifier
     */
    @Nonnull @NotEmpty public String generateIdentifier();

    /**
     * Generates an identifier.
     * 
     * @param xmlSafe true iff the result must be XML ID safe
     * @return the identifier
     */
    @Nonnull @NotEmpty public String generateIdentifier(boolean xmlSafe);
    
    /**
     * Enum of supported provider types.
     * 
     * @since 9.0.0
     */
    public enum ProviderType {
        
        /** Produces random identifiers that may not be strongly secure. */
        RANDOM,
        
        /** Produces random identifiers that rely on a theoretically secure source of randomness. */
        SECURE,
        
        /** Produces random identifiers based on type 4 UUIDs. */
        UUID
    }
    
    /**
     * Marker interface for parameters specific to particular provider types.
     * 
     * @since 9.0.0
     */
    interface ParameterSpec {
        
    }
    
    /**
     * Get an instance of a particular provider of identifiers with no parameters.
     * 
     * <p>If the type is somehow unrecognized, the {@link ProviderType#RANDOM} is used to
     * guarantee a successful result.</p>
     * 
     * @param type provider type
     * 
     * @return identifier provider
     * 
     * @since 9.0.0
     */
    @Nonnull static IdentifierGenerationStrategy getInstance(@Nonnull final ProviderType type) {
        switch (type) {
            case SECURE:
                return new SecureRandomIdentifierGenerationStrategy();
            
            case UUID:
                return new Type4UUIDIdentifierGenerationStrategy();
                
            default:
                return new RandomIdentifierGenerationStrategy();
        }
    }
    
    /**
     * Get an instance of a particular provider of identifiers with parameters.
     * 
     * @param type provider type
     * @param params implementation-specific parameter instance
     * 
     * @return identifier provider
     * 
     * @throws NoSuchAlgorithmException if the type is unknown
     * @throws InvalidAlgorithmParameterException if the parameters were invalid 
     * 
     * @since 9.0.0
     */
    @Nonnull static IdentifierGenerationStrategy getInstance(@Nonnull final ProviderType type,
            @Nonnull final ParameterSpec params) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException {
        switch (type) {
            case RANDOM:
                return new RandomIdentifierGenerationStrategy(params);
                
            case SECURE:
                return new SecureRandomIdentifierGenerationStrategy(params);
            
            case UUID:
                return new Type4UUIDIdentifierGenerationStrategy(params);
                
            default:
                throw new NoSuchAlgorithmException("Unknown IdentifierGenerationStrategy type");
        }
    }
    
}