/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.config;

import org.apache.commons.logging.*;

import java.util.Iterator;
import java.util.Locale;
import java.util.StringTokenizer;

import webwork.util.ClassLoaderUtils;

/**
 * Access to WebWork configuration. Use the constants to access defined configuration settings.
 * <p/>
 * The implementation is pluggable. The default implementation is to use the properties file "webwork.properties", which
 * must be in classpath. To install a new implementation implement {@link ConfigurationInterface} and call
 * setConfiguration() with it.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @version $Revision: 1.13 $
 */
public final class Configuration
{
    // Static --------------------------------------------------------
    static ConfigurationInterface configurationImpl;
    static ConfigurationInterface defaultImpl;
    static Locale locale; // Cached locale

    private Configuration()
    {
    }

    /**
     * Get a named setting.
     */
    public static Object get(String aName)
            throws IllegalArgumentException
    {
        Object val = getConfiguration().getImpl(aName);
        return val;
    }

    /**
     * Get a named setting as a string.
     */
    public static String getString(String aName)
            throws IllegalArgumentException
    {
        final Object o = get(aName);
        return o == null ? aName : o.toString();
    }

    /**
     * Set a named setting
     */
    public static void set(String aName, Object aValue)
            throws IllegalArgumentException, UnsupportedOperationException
    {
        getConfiguration().setImpl(aName, aValue);
    }

    /**
     * List setting names
     */
    public static Iterator list()
    {
        return getConfiguration().listImpl();
    }

    /**
     * WebWork locale accessor
     */
    public static Locale getLocale()
    {
        if (locale == null)
        {
            try
            {
                StringTokenizer localeTokens = new StringTokenizer(getString("webwork.locale"), "_");
                String lang = null;
                String country = null;
                if (localeTokens.hasMoreTokens())
                {
                    lang = localeTokens.nextToken();
                }
                if (localeTokens.hasMoreTokens())
                {
                    country = localeTokens.nextToken();
                }
                locale = new Locale(lang, country);
            }
            catch (IllegalArgumentException e)
            {
                // Default
                locale = Locale.getDefault();
            }
        }

        return locale;
    }

    /**
     * Get the current configuration implementation.
     */
    public static ConfigurationInterface getConfiguration()
    {
        return configurationImpl == null ? getDefaultConfiguration() : configurationImpl;
    }

    /**
     * Set the current configuration implementation. Can only be called once.
     */
    public static void setConfiguration(ConfigurationInterface aConfig)
            throws IllegalStateException
    {
        if (configurationImpl != null)
        {
            throw new IllegalStateException("May only set configuration implementation once");
        }

        configurationImpl = aConfig;
        locale = null; // Reset cached locale
    }

    private static ConfigurationInterface getDefaultConfiguration()
    {
        if (defaultImpl == null)
        {
            // Create bootstrap implementation
            defaultImpl = new DefaultConfiguration();

            // Create default implementation
            try
            {
                String className = getString("webwork.configuration");
                if (!className.equals(defaultImpl.getClass().getName()))
                {
                    try
                    {
                        defaultImpl = (ConfigurationInterface) ClassLoaderUtils.loadClass(className, ConfigurationInterface.class).newInstance();
                    }
                    catch (Exception e)
                    {
                        LogFactory.getLog(Configuration.class).error("Could not instantiate configuration", e);
                    }
                }
            }
            catch (IllegalArgumentException ex)
            {
                LogFactory.getLog(Configuration.class).error("No default configuration defined", ex);
            }
        }

        return defaultImpl;
    }
}
