/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.config;

import org.apache.commons.logging.LogFactory;

import webwork.util.ClassLoaderUtils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

/**
 * Default implementation of configuration. Creates and delegates to other configurations.
 *
 * @author Rickard Őberg (rickard@middleware-company.com)
 * @version $Revision: 1.14 $
 */
public class DefaultConfiguration extends AbstractConfiguration
{
    // Attributes ----------------------------------------------------
    ConfigurationInterface config;

    // Constructors --------------------------------------------------
    public DefaultConfiguration()
    {
        // Create default implementations
        // Use default properties and webwork.properties
        final List<ConfigurationInterface> configList = new ArrayList<ConfigurationInterface>();
        try
        {
            configList.add(new PropertiesConfiguration("webwork"));
        }
        catch (final Exception e)
        {
            LogFactory.getLog(this.getClass()).warn("Could not find webwork.properties");
        }
        try
        {
            configList.add(new PropertiesConfiguration("webwork/default"));
        }
        catch (final Exception e)
        {
            LogFactory.getLog(this.getClass()).error("Could not find webwork/default.properties", e);
        }

        config = new DelegatingConfiguration(configList);

        // List of configurations to delegate to
        final List<ConfigurationInterface> list = new ArrayList<ConfigurationInterface>();

        // Add list of properties configurations
        StringTokenizer configFiles = new StringTokenizer((String) config.getImpl("webwork.configuration.properties"), ",");
        while (configFiles.hasMoreTokens())
        {
            final String name = configFiles.nextToken();
            try
            {
                list.add(new PropertiesConfiguration(name));
            }
            catch (final Exception e)
            {
                LogFactory.getLog(this.getClass()).warn("Could not find " + name + ".properties. Skipping");
            }
        }

        //reset the delegating config in case we have added some
        //additional configurations that override webwork.configuration.xml
        config = new DelegatingConfiguration(list);

        // Add list of XML action configurations
        configFiles = new StringTokenizer((String) config.getImpl("webwork.configuration.xml"), ",");
        while (configFiles.hasMoreTokens())
        {
            final String name = configFiles.nextToken();
            try
            {
                list.add(new XMLActionConfiguration(name));
            }
            catch (final IllegalArgumentException e)
            {
                LogFactory.getLog(this.getClass()).warn("Skipping XML action configuration for " + name + ".xml");
            }
            catch (final Exception e)
            {
                LogFactory.getLog(this.getClass()).error("Could not create XML action configuration", e);
            }
        }

        final List configurationObjects = new ArrayList();
        try
        {
            configFiles = new StringTokenizer((String) config.getImpl("webwork.configuration.class"), ",");
            while (configFiles.hasMoreTokens())
            {
                final String name = configFiles.nextToken();
                try
                {
                    final Class<ConfigurationInterface> configurationClass = ClassLoaderUtils.loadClass(name, ConfigurationInterface.class);
                    if (configurationClass == null)
                    {
                        LogFactory.getLog(this.getClass()).warn("Could not find configuration class '" + configurationClass + "' in classpath");
                    }
                    else if (!ConfigurationInterface.class.isAssignableFrom(configurationClass))
                    {
                        LogFactory.getLog(this.getClass()).warn(
                                "ConfigurationInterface class '" + configurationClass + "' is not of type 'ConfigurationInterface'");
                    }
                    else
                    {
                        configurationObjects.add(configurationClass.newInstance());
                    }
                }
                catch (final Exception e)
                {
                    LogFactory.getLog(this.getClass()).error("Skipping configuration for class '" + name + "'", e);
                }
            }
        }
        catch (final IllegalArgumentException e)
        {
            //this means that it could not find any property 'webwork.configuration.class'
        }

        //the new configuration objects should be added at the start of the list, so that they can override any other properties
        list.addAll(0, configurationObjects);

        config = new DelegatingConfiguration(list);
    }

    /**
     * Get a named setting.
     */
    @Override
    public Object getImpl(final String aName) throws IllegalArgumentException
    {
        // Delegate
        return config.getImpl(aName);
    }

    /**
     * Set a named setting
     */
    @Override
    public void setImpl(final String aName, final Object aValue)
            throws IllegalArgumentException, UnsupportedOperationException
    {
        config.setImpl(aName, aValue);
    }

    /**
     * List setting names
     */
    @Override
    public Iterator listImpl()
    {
        return config.listImpl();
    }
}
