/*
* WebWork, Web Application Framework
*
* Distributable under Apache license.
* See terms of license at opensource.org
*/
package webwork.view.xslt;

import org.apache.commons.logging.*;
import org.xml.sax.InputSource;
import webwork.util.ServletValueStack;
import webwork.view.xslt.SAXAdapter.XMLReaderAdapter;
import webwork.view.xslt.SAXAdapter.XMLWalker;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.*;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.net.URL;

/**
 * XSL presentation servlet. Markup the result as XML and process it with an XSLT Stylesheet.
 * <p/>
 * This allows you to use XSLT stylesheets as views of WebWork actions.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @author Philipp meier (meier@o-matic.de)
 * @author Frank Febbraro (frank@phase2technology.com)
 * @version $Revision: 1.6 $
 */
public class XSLTServlet extends HttpServlet
{
    // Static final --------------------------------------------------
    public static final String STACK_NAME = "webwork.result";
    public static final String ROOT_ELEMENT_NAME = "result";

    // Protected --------------------------------------------------
    protected static Log log = LogFactory.getLog(XSLTServlet.class);

    // Private -------------------------------------------------------
    private TransformerFactory tfactory;
    private Map templatesCache;
    private XMLWalker xmlWalker;


    public void init(ServletConfig config) throws ServletException
    {
        super.init(config);

        templatesCache = new HashMap();
        xmlWalker = new XMLWalker();

        // Create a transform factory instance.
        // TODO: implement a nicer error interface.
        tfactory = TransformerFactory.newInstance();
        tfactory.setErrorListener(new ErrorListener()
        {
            public void warning(TransformerException exception) throws TransformerException
            {
                log.info("Warning: " + exception, exception);
            }

            public void error(TransformerException exception) throws TransformerException
            {
                log.error(exception);
            }

            public void fatalError(TransformerException exception) throws TransformerException
            {
                log.fatal(exception);
                throw exception;
            }
        });
    }

    public void service(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException, java.net.MalformedURLException
    {
        try
        {
            long startTime = 0;
            if (log.isDebugEnabled())
            {
                startTime = System.currentTimeMillis();
            }

            // Get root object (bean)
            Object result = ServletValueStack.getStack(request).findValue(".");
            if (result == null)
            {
                log.info("Warning: Request attribute \"" + STACK_NAME + "\" is null!");
                result = "Warning: Request attribute \"" + STACK_NAME + "\" is null!";
            }

            // Create a transformer for the stylesheet.
            Templates templates = getTemplates(request);
            Transformer transformer = templates.newTransformer();

            String mimeType = templates.getOutputProperties().getProperty(OutputKeys.MEDIA_TYPE);
            if (mimeType == null)
            {
                // guess (this is a servlet, so text/html might be the best guess)
                mimeType = "text/html";
            }
            response.setContentType(mimeType);

            // Note that in this case the XML encoding can not be processed!
            XMLReaderAdapter xmlResultAdapter = new XMLReaderAdapter(xmlWalker, result, ROOT_ELEMENT_NAME);
            Source xmlSource = new SAXSource(xmlResultAdapter, new InputSource());

            // Transform the source XML to System.out.
            PrintWriter out = response.getWriter();
            transformer.transform(xmlSource, new StreamResult(out));

            out.close(); // ...and flush...
            if (log.isDebugEnabled())
            {
                log.debug("Time:" + (System.currentTimeMillis() - startTime) + "ms");
            }
        }
        catch (TransformerConfigurationException e)
        {
            throw new ServletException("XSLT Transformation creation failed: " + e, e);
        }
        catch (TransformerException e)
        {
            throw new ServletException("XSLT Transformation failed: " + e, e);
        }
    }

    protected Templates getTemplates(HttpServletRequest request)
            throws TransformerException, java.io.IOException
    {
        String uri = (String) request.getAttribute("javax.servlet.include.servlet_path");
        if (uri == null)
        {
            uri = request.getServletPath();
        }

        if (log.isDebugEnabled())
        {
            log.debug("Template URI: " + uri);
        }

        URL templateURL = getServletContext().getResource(uri);

        if (templateURL == null)
        {
            throw new TransformerException("Stylesheet " + uri + " not found.");
        }

        Templates templates = (Templates) templatesCache.get(templateURL);
        if (templates == null)
        {
            synchronized (templatesCache)
            {
                // This may result in the template being put into the cache multiple times
                // if concurrent requests are made, but that's ok.
                log.debug("Preparing new XSLT stylesheet: " + templateURL);

                templates = tfactory.newTemplates(new StreamSource(templateURL.openStream(), templateURL.toExternalForm()));
                templatesCache.put(templateURL, templates);
            }
        }

        return templates;
    }

}
