/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import com.google.common.collect.Lists;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.PartialListing;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.hamcrest.core.StringContains;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class TestBatchedListDirectories {
    private static MiniDFSCluster cluster;
    private static Configuration conf;
    private static DistributedFileSystem dfs;
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    private static final List<Path> SUBDIR_PATHS;
    private static final List<Path> FILE_PATHS;
    private static final int FIRST_LEVEL_DIRS = 2;
    private static final int SECOND_LEVEL_DIRS = 3;
    private static final int FILES_PER_DIR = 5;
    private static final Path EMPTY_DIR_PATH;
    private static final Path DATA_FILE_PATH;
    private static final Path INACCESSIBLE_DIR_PATH;
    private static final Path INACCESSIBLE_FILE_PATH;

    private static Path getSubDirName(int i, int j) {
        return new Path(String.format("/dir%d/subdir%d", i, j));
    }

    private static Path getFileName(int i, int j, int k) {
        Path dirPath = TestBatchedListDirectories.getSubDirName(i, j);
        return new Path(dirPath, "file" + k);
    }

    private static void assertSubDirEquals(int i, int j, Path p) {
        Assert.assertTrue((boolean)p.toString().startsWith("hdfs://"));
        Path expected = TestBatchedListDirectories.getSubDirName(i, j);
        Assert.assertEquals((String)"Unexpected subdir name", (Object)expected.toString(), (Object)p.toUri().getPath());
    }

    private static void assertFileEquals(int i, int j, int k, Path p) {
        Assert.assertTrue((boolean)p.toString().startsWith("hdfs://"));
        Path expected = TestBatchedListDirectories.getFileName(i, j, k);
        Assert.assertEquals((String)"Unexpected file name", (Object)expected.toString(), (Object)p.toUri().getPath());
    }

    private static void loadData() throws Exception {
        for (int i = 0; i < 2; ++i) {
            for (int j = 0; j < 3; ++j) {
                Path dirPath = TestBatchedListDirectories.getSubDirName(i, j);
                dfs.mkdirs(dirPath);
                SUBDIR_PATHS.add(dirPath);
                for (int k = 0; k < 5; ++k) {
                    Path filePath = TestBatchedListDirectories.getFileName(i, j, k);
                    dfs.create(filePath, (short)1).close();
                    FILE_PATHS.add(filePath);
                }
            }
        }
        dfs.mkdirs(EMPTY_DIR_PATH);
        FSDataOutputStream fsout = dfs.create(DATA_FILE_PATH, (short)1);
        fsout.write(123);
        fsout.close();
        dfs.mkdirs(INACCESSIBLE_DIR_PATH);
        dfs.create(INACCESSIBLE_FILE_PATH, (short)1).close();
        dfs.setPermission(INACCESSIBLE_DIR_PATH, new FsPermission(0));
    }

    @BeforeClass
    public static void beforeClass() throws Exception {
        conf = new HdfsConfiguration();
        conf.setInt("dfs.ls.limit", 7);
        conf.setInt("dfs.batched.ls.limit", 30);
        cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
        dfs = cluster.getFileSystem();
        TestBatchedListDirectories.loadData();
    }

    @AfterClass
    public static void afterClass() {
        if (cluster != null) {
            cluster.shutdown();
        }
    }

    private static List<PartialListing<FileStatus>> getListings(List<Path> paths) throws IOException {
        ArrayList returned = Lists.newArrayList();
        RemoteIterator it = dfs.batchedListStatusIterator(paths);
        while (it.hasNext()) {
            returned.add(it.next());
        }
        return returned;
    }

    private static List<FileStatus> listingsToStatuses(List<PartialListing<FileStatus>> listings) throws IOException {
        ArrayList returned = Lists.newArrayList();
        for (PartialListing<FileStatus> listing : listings) {
            returned.addAll(listing.get());
        }
        return returned;
    }

    private static List<FileStatus> getStatuses(List<Path> paths) throws IOException {
        List<PartialListing<FileStatus>> listings = TestBatchedListDirectories.getListings(paths);
        return TestBatchedListDirectories.listingsToStatuses(listings);
    }

    @Test
    public void testEmptyPath() throws Exception {
        this.thrown.expect(FileNotFoundException.class);
        ArrayList paths = Lists.newArrayList();
        TestBatchedListDirectories.getStatuses(paths);
    }

    @Test
    public void testEmptyDir() throws Exception {
        ArrayList paths = Lists.newArrayList((Object[])new Path[]{EMPTY_DIR_PATH});
        List<PartialListing<FileStatus>> listings = TestBatchedListDirectories.getListings(paths);
        Assert.assertEquals((long)1L, (long)listings.size());
        PartialListing<FileStatus> listing = listings.get(0);
        Assert.assertEquals((Object)EMPTY_DIR_PATH, (Object)listing.getListedPath());
        Assert.assertEquals((long)0L, (long)listing.get().size());
    }

    @Test
    public void listOneFile() throws Exception {
        ArrayList paths = Lists.newArrayList();
        paths.add(FILE_PATHS.get(0));
        List<FileStatus> statuses = TestBatchedListDirectories.getStatuses(paths);
        Assert.assertEquals((long)1L, (long)statuses.size());
        TestBatchedListDirectories.assertFileEquals(0, 0, 0, statuses.get(0).getPath());
    }

    @Test
    public void listDoesNotExist() throws Exception {
        this.thrown.expect(FileNotFoundException.class);
        ArrayList paths = Lists.newArrayList();
        paths.add(new Path("/does/not/exist"));
        TestBatchedListDirectories.getStatuses(paths);
    }

    @Test
    public void listSomeDoNotExist() throws Exception {
        ArrayList paths = Lists.newArrayList();
        paths.add(new Path("/does/not/exist"));
        paths.addAll(SUBDIR_PATHS.subList(0, 2));
        paths.add(new Path("/does/not/exist"));
        paths.addAll(SUBDIR_PATHS.subList(0, 2));
        paths.add(new Path("/does/not/exist"));
        List<PartialListing<FileStatus>> listings = TestBatchedListDirectories.getListings(paths);
        for (int i = 0; i < listings.size(); ++i) {
            PartialListing<FileStatus> partial = listings.get(i);
            if (partial.getListedPath().toString().equals("/does/not/exist")) {
                try {
                    partial.get();
                    Assert.fail((String)"Expected exception");
                }
                catch (FileNotFoundException e) {
                    Assert.assertTrue((boolean)e.getMessage().contains("/does/not/exist"));
                }
                continue;
            }
            partial.get();
        }
        try {
            listings.get(listings.size() - 1).get();
            Assert.fail((String)"Expected exception");
        }
        catch (FileNotFoundException e) {
            Assert.assertTrue((boolean)e.getMessage().contains("/does/not/exist"));
        }
    }

    @Test
    public void listDirRelative() throws Exception {
        dfs.setWorkingDirectory(new Path("/dir0"));
        ArrayList paths = Lists.newArrayList((Object[])new Path[]{new Path(".")});
        List<FileStatus> statuses = TestBatchedListDirectories.getStatuses(paths);
        Assert.assertEquals((String)"Wrong number of items", (long)3L, (long)statuses.size());
        for (int i = 0; i < 3; ++i) {
            FileStatus stat = statuses.get(i);
            TestBatchedListDirectories.assertSubDirEquals(0, i, stat.getPath());
        }
    }

    @Test
    public void listFilesRelative() throws Exception {
        dfs.setWorkingDirectory(new Path("/dir0"));
        ArrayList paths = Lists.newArrayList((Object[])new Path[]{new Path("subdir0")});
        List<FileStatus> statuses = TestBatchedListDirectories.getStatuses(paths);
        Assert.assertEquals((String)"Wrong number of items", (long)5L, (long)statuses.size());
        for (int i = 0; i < 5; ++i) {
            FileStatus stat = statuses.get(i);
            TestBatchedListDirectories.assertFileEquals(0, 0, i, stat.getPath());
        }
    }

    @Test
    public void testDFSHasCapability() throws Throwable {
        Assert.assertTrue((String)"FS does not declare PathCapability support", (boolean)dfs.hasPathCapability(new Path("/"), "fs.capability.batch.listing"));
    }

    private void listFilesInternal(int numFiles) throws Exception {
        List<Path> paths = FILE_PATHS.subList(0, numFiles);
        List<FileStatus> statuses = TestBatchedListDirectories.getStatuses(paths);
        Assert.assertEquals((long)paths.size(), (long)statuses.size());
        for (int i = 0; i < paths.size(); ++i) {
            Path p = paths.get(i);
            FileStatus stat = statuses.get(i);
            Assert.assertEquals((Object)p.toUri().getPath(), (Object)stat.getPath().toUri().getPath());
        }
    }

    @Test
    public void listOneFiles() throws Exception {
        this.listFilesInternal(1);
    }

    @Test
    public void listSomeFiles() throws Exception {
        this.listFilesInternal(FILE_PATHS.size() / 2);
    }

    @Test
    public void listAllFiles() throws Exception {
        this.listFilesInternal(FILE_PATHS.size());
    }

    private void listDirectoriesInternal(int numDirs) throws Exception {
        List<Path> paths = SUBDIR_PATHS.subList(0, numDirs);
        List<PartialListing<FileStatus>> listings = TestBatchedListDirectories.getListings(paths);
        LinkedHashMap<Path, ArrayList> listing = new LinkedHashMap<Path, ArrayList>();
        for (PartialListing<FileStatus> partialListing : listings) {
            Path parent = partialListing.getListedPath();
            if (!listing.containsKey(parent)) {
                listing.put(parent, Lists.newArrayList());
            }
            ((List)listing.get(parent)).addAll(partialListing.get());
        }
        Assert.assertEquals((long)paths.size(), (long)listing.size());
        int pathIdx = 0;
        for (Map.Entry entry : listing.entrySet()) {
            Path expected = paths.get(pathIdx++);
            Path parent = (Path)entry.getKey();
            List children = (List)entry.getValue();
            Assert.assertEquals((Object)expected, (Object)parent);
            Assert.assertEquals((long)5L, (long)children.size());
        }
    }

    @Test
    public void listOneDirectory() throws Exception {
        this.listDirectoriesInternal(1);
    }

    @Test
    public void listSomeDirectories() throws Exception {
        this.listDirectoriesInternal(SUBDIR_PATHS.size() / 2);
    }

    @Test
    public void listAllDirectories() throws Exception {
        this.listDirectoriesInternal(SUBDIR_PATHS.size());
    }

    @Test
    public void listTooManyDirectories() throws Exception {
        this.thrown.expect(RemoteException.class);
        this.thrown.expectMessage(StringContains.containsString((String)"Too many source paths"));
        ArrayList paths = Lists.newArrayList(FILE_PATHS);
        paths.add(SUBDIR_PATHS.get(0));
        TestBatchedListDirectories.getStatuses(paths);
    }

    @Test
    public void listDirsAndEmpty() throws Exception {
        ArrayList paths = Lists.newArrayList();
        paths.add(EMPTY_DIR_PATH);
        paths.add(FILE_PATHS.get(0));
        paths.add(EMPTY_DIR_PATH);
        List<PartialListing<FileStatus>> listings = TestBatchedListDirectories.getListings(paths);
        Assert.assertEquals((long)3L, (long)listings.size());
        Assert.assertEquals((long)0L, (long)listings.get(0).get().size());
        Assert.assertEquals((long)1L, (long)listings.get(1).get().size());
        Assert.assertEquals((Object)FILE_PATHS.get(0).toString(), (Object)((FileStatus)listings.get(1).get().get(0)).getPath().toUri().getPath());
        Assert.assertEquals((long)0L, (long)listings.get(2).get().size());
    }

    @Test
    public void listSamePaths() throws Exception {
        int i;
        ArrayList paths = Lists.newArrayList();
        paths.add(SUBDIR_PATHS.get(0));
        paths.add(SUBDIR_PATHS.get(0));
        paths.add(FILE_PATHS.get(0));
        paths.add(FILE_PATHS.get(0));
        List<FileStatus> statuses = TestBatchedListDirectories.getStatuses(paths);
        Assert.assertEquals((long)12L, (long)statuses.size());
        List<FileStatus> slice = statuses.subList(0, 5);
        for (i = 0; i < 5; ++i) {
            TestBatchedListDirectories.assertFileEquals(0, 0, i, slice.get(i).getPath());
        }
        slice = statuses.subList(5, 10);
        for (i = 0; i < 5; ++i) {
            TestBatchedListDirectories.assertFileEquals(0, 0, i, slice.get(i).getPath());
        }
        TestBatchedListDirectories.assertFileEquals(0, 0, 0, statuses.get(10).getPath());
        TestBatchedListDirectories.assertFileEquals(0, 0, 0, statuses.get(11).getPath());
    }

    @Test
    public void listLocatedStatus() throws Exception {
        ArrayList paths = Lists.newArrayList();
        paths.add(DATA_FILE_PATH);
        RemoteIterator it = dfs.batchedListLocatedStatusIterator((List)paths);
        PartialListing listing = (PartialListing)it.next();
        List statuses = listing.get();
        Assert.assertEquals((long)1L, (long)statuses.size());
        Assert.assertTrue((((LocatedFileStatus)statuses.get(0)).getBlockLocations().length > 0 ? 1 : 0) != 0);
    }

    private void listAsNormalUser(final List<Path> paths) throws IOException, InterruptedException {
        UserGroupInformation ugi = UserGroupInformation.createRemoteUser((String)"tiffany");
        ugi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Void>(){

            @Override
            public Void run() throws Exception {
                DistributedFileSystem fs = (DistributedFileSystem)FileSystem.get((URI)cluster.getURI(), (Configuration)conf);
                RemoteIterator it = fs.batchedListStatusIterator(paths);
                PartialListing listing = (PartialListing)it.next();
                listing.get();
                return null;
            }
        });
    }

    @Test
    public void listInaccessibleDir() throws Exception {
        this.thrown.expect(AccessControlException.class);
        ArrayList paths = Lists.newArrayList((Object[])new Path[]{INACCESSIBLE_DIR_PATH});
        this.listAsNormalUser(paths);
    }

    @Test
    public void listInaccessibleFile() throws Exception {
        this.thrown.expect(AccessControlException.class);
        ArrayList paths = Lists.newArrayList((Object[])new Path[]{INACCESSIBLE_FILE_PATH});
        this.listAsNormalUser(paths);
    }

    static {
        SUBDIR_PATHS = Lists.newArrayList();
        FILE_PATHS = Lists.newArrayList();
        EMPTY_DIR_PATH = new Path("/emptydir");
        DATA_FILE_PATH = new Path("/datafile");
        INACCESSIBLE_DIR_PATH = new Path("/noperms");
        INACCESSIBLE_FILE_PATH = new Path(INACCESSIBLE_DIR_PATH, "nopermsfile");
    }
}

