/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.examples;

import com.google.common.collect.Lists;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;
import java.util.List;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Peer;
import org.bitcoinj.core.PeerGroup;
import org.bitcoinj.core.listeners.PeerConnectedEventListener;
import org.bitcoinj.core.listeners.PeerDisconnectedEventListener;
import org.bitcoinj.net.discovery.DnsDiscovery;
import org.bitcoinj.net.discovery.PeerDiscovery;
import org.bitcoinj.params.MainNetParams;
import org.bitcoinj.utils.BriefLogFormatter;

public class PeerMonitor {
    private NetworkParameters params;
    private PeerGroup peerGroup;
    private PeerTableModel peerTableModel;
    private PeerTableRenderer peerTableRenderer;
    private final HashMap<Peer, String> reverseDnsLookups = new HashMap();

    public static void main(String[] args) throws Exception {
        BriefLogFormatter.init();
        new PeerMonitor();
    }

    public PeerMonitor() {
        this.setupNetwork();
        this.setupGUI();
        this.peerGroup.startAsync();
    }

    private void setupNetwork() {
        this.params = MainNetParams.get();
        this.peerGroup = new PeerGroup(this.params, null);
        this.peerGroup.setUserAgent("PeerMonitor", "1.0");
        this.peerGroup.setMaxConnections(4);
        this.peerGroup.addPeerDiscovery((PeerDiscovery)new DnsDiscovery(this.params));
        this.peerGroup.addConnectedEventListener(new PeerConnectedEventListener(){

            public void onPeerConnected(Peer peer, int peerCount) {
                PeerMonitor.this.refreshUI();
                PeerMonitor.this.lookupReverseDNS(peer);
            }
        });
        this.peerGroup.addDisconnectedEventListener(new PeerDisconnectedEventListener(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onPeerDisconnected(Peer peer, int peerCount) {
                PeerMonitor.this.refreshUI();
                HashMap hashMap = PeerMonitor.this.reverseDnsLookups;
                synchronized (hashMap) {
                    PeerMonitor.this.reverseDnsLookups.remove(peer);
                }
            }
        });
    }

    private void lookupReverseDNS(final Peer peer) {
        new Thread(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                String reverseDns = peer.getAddress().getAddr().getCanonicalHostName();
                HashMap hashMap = PeerMonitor.this.reverseDnsLookups;
                synchronized (hashMap) {
                    PeerMonitor.this.reverseDnsLookups.put(peer, reverseDns);
                }
                PeerMonitor.this.refreshUI();
            }
        }.start();
    }

    private void refreshUI() {
        SwingUtilities.invokeLater(new Runnable(){

            @Override
            public void run() {
                PeerMonitor.this.peerTableModel.updateFromPeerGroup();
            }
        });
    }

    private void setupGUI() {
        JFrame window = new JFrame("Network monitor");
        window.setDefaultCloseOperation(0);
        window.addWindowListener(new WindowAdapter(){

            @Override
            public void windowClosing(WindowEvent windowEvent) {
                System.out.println("Shutting down ...");
                PeerMonitor.this.peerGroup.stop();
                System.out.println("Shutdown complete.");
                System.exit(0);
            }
        });
        JPanel panel = new JPanel();
        JLabel instructions = new JLabel("Number of peers to connect to: ");
        final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(4, 0, 100, 1);
        spinnerModel.addChangeListener(new ChangeListener(){

            @Override
            public void stateChanged(ChangeEvent changeEvent) {
                PeerMonitor.this.peerGroup.setMaxConnections(spinnerModel.getNumber().intValue());
            }
        });
        JSpinner numPeersSpinner = new JSpinner(spinnerModel);
        panel.add(instructions);
        panel.add(numPeersSpinner);
        window.getContentPane().add((Component)panel, "North");
        this.peerTableModel = new PeerTableModel();
        JTable peerTable = new JTable(this.peerTableModel);
        peerTable.setAutoCreateRowSorter(true);
        this.peerTableRenderer = new PeerTableRenderer(this.peerTableModel);
        peerTable.setDefaultRenderer(String.class, this.peerTableRenderer);
        peerTable.setDefaultRenderer(Integer.class, this.peerTableRenderer);
        peerTable.setDefaultRenderer(Long.class, this.peerTableRenderer);
        peerTable.getColumnModel().getColumn(0).setPreferredWidth(300);
        JScrollPane scrollPane = new JScrollPane(peerTable);
        window.getContentPane().add((Component)scrollPane, "Center");
        window.pack();
        window.setSize(720, 480);
        window.setVisible(true);
        new Timer(1000, new ActionListener(){

            @Override
            public void actionPerformed(ActionEvent actionEvent) {
                PeerMonitor.this.peerTableModel.updateFromPeerGroup();
            }
        }).start();
    }

    private class PeerTableRenderer
    extends JLabel
    implements TableCellRenderer {
        private final PeerTableModel model;
        private final Font normal;
        private final Font bold;

        public PeerTableRenderer(PeerTableModel model) {
            this.model = model;
            this.normal = new Font("Sans Serif", 0, 12);
            this.bold = new Font("Sans Serif", 1, 12);
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object contents, boolean selected, boolean hasFocus, int row, int column) {
            boolean isPingColumn;
            row = table.convertRowIndexToModel(row);
            column = table.convertColumnIndexToModel(column);
            String str = contents.toString();
            if (this.model.connectedPeers == null || this.model.pendingPeers == null) {
                this.setText(str);
                return this;
            }
            if (row >= this.model.connectedPeers.size()) {
                this.setFont(this.normal);
                this.setForeground(Color.LIGHT_GRAY);
            } else {
                long height;
                if (this.model.connectedPeers.get(row) == PeerMonitor.this.peerGroup.getDownloadPeer()) {
                    this.setFont(this.bold);
                } else {
                    this.setFont(this.normal);
                }
                this.setForeground(Color.BLACK);
                if (column == 3 && (height = ((Long)contents).longValue()) != (long)PeerMonitor.this.peerGroup.getMostCommonChainHeight()) {
                    str = height + " \u2022 ";
                }
            }
            boolean bl = isPingColumn = column == 4 || column == 5;
            if (isPingColumn && contents.equals(Long.MAX_VALUE)) {
                str = "";
            }
            this.setText(str);
            return this;
        }
    }

    private class PeerTableModel
    extends AbstractTableModel {
        public static final int IP_ADDRESS = 0;
        public static final int PROTOCOL_VERSION = 1;
        public static final int USER_AGENT = 2;
        public static final int CHAIN_HEIGHT = 3;
        public static final int PING_TIME = 4;
        public static final int LAST_PING_TIME = 5;
        public List<Peer> connectedPeers = Lists.newArrayList();
        public List<Peer> pendingPeers = Lists.newArrayList();

        private PeerTableModel() {
        }

        public void updateFromPeerGroup() {
            this.connectedPeers = PeerMonitor.this.peerGroup.getConnectedPeers();
            this.pendingPeers = PeerMonitor.this.peerGroup.getPendingPeers();
            this.fireTableDataChanged();
        }

        @Override
        public int getRowCount() {
            return this.connectedPeers.size() + this.pendingPeers.size();
        }

        @Override
        public String getColumnName(int i) {
            switch (i) {
                case 0: {
                    return "Address";
                }
                case 1: {
                    return "Protocol version";
                }
                case 2: {
                    return "User Agent";
                }
                case 3: {
                    return "Chain height";
                }
                case 4: {
                    return "Average ping";
                }
                case 5: {
                    return "Last ping";
                }
            }
            throw new RuntimeException();
        }

        @Override
        public int getColumnCount() {
            return 6;
        }

        @Override
        public Class<?> getColumnClass(int column) {
            switch (column) {
                case 1: {
                    return Integer.class;
                }
                case 3: 
                case 4: 
                case 5: {
                    return Long.class;
                }
            }
            return String.class;
        }

        @Override
        public Object getValueAt(int row, int col) {
            if (row >= this.connectedPeers.size()) {
                Peer peer = this.pendingPeers.get(row - this.connectedPeers.size());
                switch (col) {
                    case 0: {
                        return this.getAddressForPeer(peer);
                    }
                    case 1: {
                        return 0;
                    }
                    case 3: 
                    case 4: 
                    case 5: {
                        return 0L;
                    }
                }
                return "(pending)";
            }
            Peer peer = this.connectedPeers.get(row);
            switch (col) {
                case 0: {
                    return this.getAddressForPeer(peer);
                }
                case 1: {
                    return Integer.toString(peer.getPeerVersionMessage().clientVersion);
                }
                case 2: {
                    return peer.getPeerVersionMessage().subVer;
                }
                case 3: {
                    return peer.getBestHeight();
                }
                case 4: 
                case 5: {
                    return col == 4 ? peer.getPingTime() : peer.getLastPingTime();
                }
            }
            throw new RuntimeException();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Object getAddressForPeer(Peer peer) {
            String s;
            HashMap hashMap = PeerMonitor.this.reverseDnsLookups;
            synchronized (hashMap) {
                s = (String)PeerMonitor.this.reverseDnsLookups.get(peer);
            }
            if (s != null) {
                return s;
            }
            return peer.getAddress().getAddr().getHostAddress();
        }
    }
}

