/* Simple JavaScript Inheritance
 * By John Resig http://ejohn.org/
 * MIT Licensed.
 */
// Inspired by base2 and Prototype
(function(){
    var initializing = false, fnTest = /xyz/.test(function(){xyz;}) ? /\b_super\b/ : /.*/;

    // The base Class implementation (does nothing)
    this.Class = function(){};

    // Create a new Class that inherits from this class
    Class.extend = function(prop) {
        var _super = this.prototype;

        // Instantiate a base class (but only create the instance,
        // don't run the init constructor)
        initializing = true;
        var prototype = new this();
        initializing = false;

        // Copy the properties over onto the new prototype
        for (var name in prop) {
            // Check if we're overwriting an existing function
            prototype[name] = typeof prop[name] == "function" &&
            typeof _super[name] == "function" && fnTest.test(prop[name]) ?
                    (function(name, fn){
                        return function() {
                            var tmp = this._super;

                            // Add a new ._super() method that is the same method
                            // but on the super-class
                            this._super = _super[name];

                            // The method only need to be bound temporarily, so we
                            // remove it when we're done executing
                            var ret = fn.apply(this, arguments);
                            this._super = tmp;

                            return ret;
                        };
                    })(name, prop[name]) :
                    prop[name];
        }

        // The dummy class constructor
        function Class() {
            // All construction is actually done in the init method
            if ( !initializing && this.init )
                this.init.apply(this, arguments);
        }

        // Populate our constructed prototype object
        Class.prototype = prototype;

        // Enforce the constructor to be what we expect
        Class.prototype.constructor = Class;

        // And make this class extendable
        Class.extend = arguments.callee;

        return Class;
    };
})();
/**
 * jQuery-Plugin "butterHandleAutoComplete" for text autocomplete tag. Initializes auto complete functionality to
 * text component.
 *
 * How to use:
 * jQuery("#selector")._butterAutoCompleteInit();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------

    $.fn._butterAutoCompleteInit = function () {
        return this.each(function () {
            new AutocompleteList(this);
        });
    };

    // define objects --------------------------------------------------------------------

    var AutocompleteList = Class.extend({
        init: function (rootElement) {
            this.SEARCH_REQUEST_DELAY = 300;// in ms

            var $autocompleteTmp = $(rootElement);
            this.$input = $autocompleteTmp.prev();
            this.$input.parent().css({position: "relative"});
            this.autocompleteId = $autocompleteTmp.attr("id");
            this.$selectedOption = null;
            this.ignoreKeyupEvent = false;
            this.requestDelayTimerId = null;
            this.isRequestRunning = false;
            this.areChangesMadeWhileRequestWasRunning = false;

            this._keyCodes = {
                //backspace: 8,
                tab: 9,
                enter: 13,
                shift: 16,
                ctrl: 17,
                alt: 18,
                pause: 19,
                caps_lock: 20,
                escape: 27,
                page_up: 33,
                page_down: 34,
                end: 35,
                home: 36,
                arrow_left: 37,
                arrow_up: 38,
                arrow_right: 39,
                arrow_down: 40,
                insert: 45,
                // 'delete' is a reserved key word
                delete_key: 46,
                left_window_key: 91,
                right_window_key: 92,
                select_key: 93,
                num_lock: 144,
                scroll_lock: 145
            };

            var self = this;
            self.$input
                    .on("keydown", function (event) {
                        if (event.which === self._keyCodes.enter) {
                            self._handleEnterKeyDown(event);
                        } else if (event.which === self._keyCodes.arrow_up
                                || event.which === self._keyCodes.arrow_down) {
                            self._handleArrowUpAndDownKeyDown(event);
                        } else if (event.which === self._keyCodes.escape) {
                            self._handleEscapeKeyDown(event);
                        }
                    })
                    .on("keyup", function (event) {
                        // don't handle other keys than character keys
                        for (keyName in self._keyCodes) {
                            if (self._keyCodes[keyName] === event.which) {
                                self._stopEvent(event);
                                return;
                            }
                        }

                        if (self.ignoreKeyupEvent) {
                            self._stopEvent(event);
                            self.ignoreKeyupEvent = false;
                            return;
                        }

                        if (self.$input.val().length === 0) {
                            self._stopEvent(event);
                            self._hideAutocompleteResultList();
                            return;
                        }

                        self._trySendJsfAjaxRequest();
                    })
                    .on("blur", function (event) {
                        window.setTimeout(function () {
                            self._hideAutocompleteResultList();
                        }, 100);
                    });
        },

        _handleEnterKeyDown: function (event) {
            if (this.$selectedOption !== null) {
                this._stopEvent(event);
                this._setSelectedValue();
            }
        },

        _handleArrowUpAndDownKeyDown: function (event) {
            this._stopEvent(event);
            var $autocomplete = this._getAutocompleteElement();
            if (!$autocomplete.is(":visible") && this.$input.val().length > 0) {
                this._trySendJsfAjaxRequest();
            } else if ($autocomplete.is(":visible") && $autocomplete.find("li").length > 0) {
                if (this.$selectedOption === null) {
                    this._selectResultOptionElement($autocomplete.find("li")[0]);
                } else {
                    this._moveResultOptionElementSelectionCursor(
                            $autocomplete, event.which === this._keyCodes.arrow_up ? -1 : 1);
                }
            }
        },

        _handleEscapeKeyDown: function (event) {
            this._stopEvent(event);
            this._hideAutocompleteResultList();
        },

        _trySendJsfAjaxRequest: function () {
            var self = this;

            if (self.isRequestRunning) {
                // console.log("request is active, so remember that changes has been made while request was running");
                self.areChangesMadeWhileRequestWasRunning = true;
            }

            if (self.requestDelayTimerId !== null) {
                window.clearTimeout(self.requestDelayTimerId)
            }

            self.requestDelayTimerId = window.setTimeout(function () {
                self.requestDelayTimerId = null;
                self._sendJsfAjaxRequest();
            }, self.SEARCH_REQUEST_DELAY);
        },

        _sendJsfAjaxRequest: function () {
            var self = this;

            if (self.isRequestRunning) {
                // console.log("request is running, abort");
                return;
            }
            self.isRequestRunning = true;

            self.areChangesMadeWhileRequestWasRunning = false;
            self._showLoadingSpinner();

            // console.log("starting request");

            var id = self.$input.parent().parent().attr('id');

            jsf.ajax.request(id, "autocomplete", {
                "javax.faces.behavior.event": "autocomplete",
                render: self.autocompleteId,
                params: self.$input.val(),
                onevent: function (data) {
                    if (data.status === "success") {
                        // console.log("request finished");

                        // only show result if input field still has focus
                        if (self.$input.is(":focus")) {
                            self._handleAutocompleteResultListVisibility();
                        }
                        self._hideLoadingSpinner();
                        self.isRequestRunning = false;

                        if (self.areChangesMadeWhileRequestWasRunning) {
                            // console.log("changes made while request was running, start new request automatically");
                            self._sendJsfAjaxRequest();
                        }
                    }
                }
            });
        },

        _handleAutocompleteResultListVisibility: function () {
            var self = this;
            var $autocomplete = self._getAutocompleteElement();

            if ($autocomplete.find("li").length > 0) {
                self._initAndShowAutocompleteResultList();
            } else {
                self._hideAutocompleteResultList();
            }
        },

        _showLoadingSpinner: function () {
            $('<div class="butter-dropdownlist-spinner-container"><div class="butter-dropdownlist-spinner"></div></div>')
                    .appendTo(this.$input.parent());
        },

        _hideLoadingSpinner: function () {
            this.$input.parent().find(".butter-dropdownlist-spinner").remove();
        },

        _initAndShowAutocompleteResultList: function () {
            var self = this;
            var $autocomplete = self._getAutocompleteElement();
            $autocomplete
                    .show()
                    .css({
                        width: self.$input.innerWidth()
                    })
                    .highlight(self.$input.val());

            $autocomplete.find("ul")
                    .on("mouseleave", function () {
                        self._clearResultOptionSelection();
                    });

            $autocomplete.find("li")
                    .on("mousedown", function () {
                        self._setSelectedValue();
                    })
                    .on("mouseenter", function () {
                        self._selectResultOptionElement(this);
                    });
        },

        _selectResultOptionElement: function (optionElement) {
            this._clearResultOptionSelection();
            var $selectedOptionElement = $(optionElement);
            $selectedOptionElement.addClass("butter-dropdownlist-resultItem-selected");
            this.$selectedOption = $selectedOptionElement;
        },

        _clearResultOptionSelection: function () {
            this.$selectedOption = null;
            this._getAutocompleteElement()
                    .find(".butter-dropdownlist-resultItem-selected")
                    .removeClass("butter-dropdownlist-resultItem-selected");
        },

        _moveResultOptionElementSelectionCursor: function ($autocomplete, direction) {
            if (direction > 0) {
                var $next = this.$selectedOption.next();
                if ($next.length > 0) {
                    this._selectResultOptionElement($next[0]);
                } else {
                    //there is no next
                    this._selectResultOptionElement($autocomplete.find("li")[0]);
                }
            } else {
                var $prev = this.$selectedOption.prev();
                if ($prev.length > 0) {
                    this._selectResultOptionElement($prev[0]);
                } else {
                    //there is no previous
                    var resultListOptions = $autocomplete.find("li");
                    this._selectResultOptionElement(resultListOptions[resultListOptions.length - 1]);
                }
            }
        },

        _setSelectedValue: function () {
            if (this.$selectedOption !== null) {
                this.ignoreKeyupEvent = true;
                this.$input
                        .val(this.$selectedOption.attr("data-select-value"))
                        .change()
                        .focus()
                        .keyup();
                this._hideAutocompleteResultList();
            }
        },

        _hideAutocompleteResultList: function () {
            if (this.requestDelayTimerId !== null) {
                window.clearTimeout(this.requestDelayTimerId)
            }
            this.$selectedOption = null;
            this._getAutocompleteElement().hide();
        },

        _getAutocompleteElement: function () {
            return $(document.getElementById(this.autocompleteId));
        },

        _stopEvent: function (event) {
            event.stopPropagation();
            event.preventDefault();
        }
    });

}(jQuery));
/**
 * jQuery-Plugin to handle bootstrap fixes.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#someTreeSelector").fixBootstrapDropDown();
 */
(function ($) {
    $.fn.fixBootstrapDropDown = function () {

        return this.each(function () {
            $('.dropdown-menu').on('click', function(e) {
                if($(this).hasClass('dropdown-menu-form')) {
                    e.stopPropagation();
                }
            });
        });

    };
}(jQuery));
/**
 * butterItemFilterField is a jQuery plugin that filters html element with the css class <code>filterable-item</code>.
 * It is applied to the search field.<br/>
 * If no filter text is entered, then all filterable-items are displayed. Else the search field value is matched against <b>all</b> text contained by a filterable-item.
 *
 * How to use:
 * jQuery("#someInputSelector").butterItemFilterField();
 *
 * Author: Yann Massard
 */
(function ($) {
    var delay = (function () {
        var timer = 0;
        return function (callback, ms) {
            clearTimeout(timer);
            timer = setTimeout(callback, ms);
        };
    })();

    // extend jQuery --------------------------------------------------------------------
    $.fn.butterItemFilterField = function (filterableItemContainerSelector) {
        return this.each(function () {
            var $this = $(this);
            $this.keyup(function () {
                delay(function () {
                    var filterValue = $this.val();

                    // find container again every time, because it could have been rerendered.
                    var $filterableItemContainer;
                    if (filterableItemContainerSelector) {
                        $filterableItemContainer = $(filterableItemContainerSelector);
                    } else {
                        var containerSelector = $this.attr('data-filterable-item-container');
                        $filterableItemContainer = $(containerSelector);
                    }

                    var filterableItems = $filterableItemContainer.find('.filterable-item');

                    filterableItems.each(function (i, elem) {
                        var $filterableItem = $(elem);
                        if ($filterableItem.is(':containsIgnoreCase(' + filterValue + ')')) {
                            $filterableItem.removeAttr("hidden");
                            $filterableItem.highlight(filterValue);
                        } else {
                            $filterableItem.attr("hidden", "hidden");
                        }

                    });
                }, 300);
            });
        });
    };
}(jQuery));

(function ($) {
    $.expr[":"].containsIgnoreCase = $.expr.createPseudo(function (arg) {
        return function (elem) {
            return !arg || $(elem).text().toUpperCase().indexOf(arg.toUpperCase()) >= 0;
        };
    });
}(jQuery));
/**
 * jQuery-Plugin "Animate dots" for tree animation. Animates a waiting dot line (4 dots) in an interval of 200 millis
 * as html string in given component. Note: existing html code will be cleared.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#selector").startDots();
 * jQuery("#selector").stopDots();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------

    var intervalTrigger = null;

    $.fn.startDots = function () {
        return this.each(function () {
            var $originalElement = $(this);

            $originalElement.html('');

            intervalTrigger = setInterval(function () {
                $originalElement.append('.');

                if ($originalElement.html().length > 5) {
                    $originalElement.html('');
                }
            }, 200);
        });
    };

    $.fn.stopDots = function () {
        return this.each(function () {
            var $originalElement = $(this);

            $originalElement.html('');
            window.clearInterval(intervalTrigger);
        });
    };

}(jQuery));
(function ($) {
    $.fn.highlight = function (searchString) {
        var highlightClassName = "search-highlighted";
        var regex = new RegExp(searchString, "gi");

        var elements = this.find('*');
        this.each(function () {
            elements.push(this);
            $(this).find('.' + highlightClassName).contents().unwrap();
            this.normalize();
        });

        return elements.each(function () {
            var $this = $(this);

            if (searchString && searchString !== '') {
                $this.contents()
                    .filter(function () {
                        return this.nodeType === 3 && regex.test(this.nodeValue);
                    })
                    .replaceWith(function () {
                        return (this.nodeValue || "").replace(regex, function (match) {
                            return "<span class=\"" + highlightClassName + "\">" + match + "</span>";
                        });
                    });
            }
        });
    };
}(jQuery));
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.markdownReadonly = function () {
        var root = $(this);

        var $readonlyMarkdown = root.find('.butter-component-value-readonly-wrapper');
        var markdownText = $readonlyMarkdown.html()
            .replace('&amp;', '&')
            .replace('&lt;', '<')
            .replace('&gt;', '>');
        var markdownTextToHtml = new showdown.Converter().makeHtml(markdownText);

        $readonlyMarkdown.empty();
        $readonlyMarkdown.append(markdownTextToHtml);
    };
}(jQuery));
if (typeof butter === 'undefined') {
    butter = {};
}
butter.modal = {};

butter.modal.open = function (/* string */ modalPanelId) {
    // console.log('Opening modal panel with data-modal-id ' + modalPanelId);
    $('.butter-modal[data-modal-id=' + modalPanelId + ']').modal({show: true, keyboard: false, backdrop: 'static'})
};

butter.modal.close = function (/* string */ modalPanelId) {
    // console.log('Closing modal panel with data-modal-id ' + modalPanelId);
    $('.butter-modal[data-modal-id=' + modalPanelId + ']').modal('hide');
};
/**
 * jQuery-Plugin "Number Spinner" for input fields.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("someButterComponentSelector").butterNumberSpinner();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.butterNumberSpinner = function (options) {
        return this.each(function () {
            new NumberSpinner(this, options);
        });
    };

    // define classes --------------------------------------------------------------------
    var NumberSpinner = Class.extend({
        init: function (element, options) {
            this.$input = $(element).find("input");
            this._initInput();
            this._initOptions(options);
            this._counter = null;
            this.setCounter(0);
            this._isValueSet = false;
            this._initButtons();
            this._initArrowKeys();
            this._initMouseWheel();
        },

        _initInput: function () {
            var self = this;
            this.$input
                .addClass("butter-component-number-input")
                .blur(function () {
                    self._setValueOnBlur();
                })
                .parent().addClass("input-group");
        },

        _initOptions: function (options) {
            var defaultOptions = {
                step: 1,
                disabled: false
            };
            this._options = $.extend({}, defaultOptions, options);

            // ensure that this values are numbers
            if (this._options.step !== undefined) {
                this._options.step = this._options.step * 1;
            }
            if (this._options.min !== undefined) {
                this._options.min = this._options.min * 1;
            }
            if (this._options.max !== undefined) {
                this._options.max = this._options.max * 1;
            }
        },

        _initButtons: function () {
            var $outerAddon = $("<div>")
                .addClass("input-group-append")
                .insertAfter(this.$input);

            var $addon = $("<span>")
                .addClass("input-group-text")
                .addClass("butter-component-number-buttons")
                .appendTo($outerAddon);

            var self = this;
            $("<span>")
                .addClass("glyphicon glyphicon-chevron-up")
                .addClass("butter-component-number-button")
                .addClass(function () {
                    return self._options.disabled ? "disabled" : "";
                })
                .click(function () {
                    if (!self._options.disabled) {
                        self.increaseCounter();
                    }
                })
                .appendTo($addon);

            $("<span>")
                .addClass("glyphicon glyphicon-chevron-down")
                .addClass("butter-component-number-button")
                .addClass(function () {
                    return self._options.disabled ? "disabled" : "";
                })
                .click(function () {
                    if (!self._options.disabled) {
                        self.decreaseCounter();
                    }
                })
                .appendTo($addon);
        },

        _initArrowKeys: function () {
            var self = this;
            this.$input.keydown(function (event) {
                if (event.which === 38) {
                    event.stopPropagation();
                    event.preventDefault();
                    self.increaseCounter();
                } else if (event.which === 40) {
                    event.stopPropagation();
                    event.preventDefault();
                    self.decreaseCounter();
                }
            });
        },

        _initMouseWheel: function () {
            var self = this;
            this.$input.on("mousewheel DOMMouseScroll", function (event) {
                if (!self.$input.is(':focus')) {
                    return;
                }

                var delta = event.originalEvent.wheelDelta || -event.originalEvent.deltaY || -event.originalEvent.detail;

                event.stopPropagation();
                event.preventDefault();

                if (delta < 0) {
                    self.decreaseCounter();
                } else {
                    self.increaseCounter();
                }
            });
        },

        _setValueOnBlur: function () {
            var value = this.$input.val();
            if (this._isStringEmpty(value)) {
                this.$input.val("");
                this.setCounter(0);
                this._isValueSet = false;
            } else {
                var parsedInt = parseInt(value);
                if (isNaN(parsedInt)) {
                    if (this._isValueSet) {
                        this.$input.val(this._counter);
                    } else {
                        this.$input.val("");
                        this.setCounter(0);
                        this._isValueSet = false;
                    }
                } else {
                    this.setCounter(parsedInt);
                    this.$input.val(this._counter);
                    this._isValueSet = true;
                }
            }
        },

        increaseCounter: function () {
            if (this._isValueSet) {
                this.setCounter(this._counter + this._options.step);
            } else {
                this._isValueSet = true;
            }
            this.$input.val(this._counter);
            this.$input.change();
        },

        decreaseCounter: function () {
            if (this._isValueSet) {
                this.setCounter(this._counter - this._options.step);
            } else {
                this._isValueSet = true;
            }
            this.$input.val(this._counter);
            this.$input.change();
        },

        _isStringEmpty: function (value) {
            return (value.length === 0 || !value.trim());
        },

        setCounter: function (value) {
            if (this._options.min !== undefined && value < this._options.min) {
                this._counter = this._options.min;
            } else if (this._options.max !== undefined && value > this._options.max) {
                this._counter = this._options.max;
            } else {
                this._counter = value;
            }
        }
    });
}(jQuery));
if (typeof butter === 'undefined') {
    butter = {};
}
butter.prettyprint = {};

/**
 * calls pretty print javascript framework and removes first and last empty children.
 */
butter.prettyprint.cleanup = function() {
    prettyPrint();

    jQuery('.butter-component-prettyprint').each(function () {
        var $firstPreChild = jQuery(this).find("pre span").first();
        var $lastPreChild = jQuery(this).find("pre span").last();


        if (!(typeof $firstPreChild.html() === "undefined")) {
            if (!$firstPreChild.html().trim()) {
                $firstPreChild.remove();
            }
        }
        if (!(typeof $lastPreChild.html() === "undefined")) {
            if (!$lastPreChild.html().trim()) {
                $lastPreChild.remove();
            }
        }
    });
};
/**
 * How to use:
 * jQuery("#someSelector").scrollToFirst() or jQuery("#someSelector").scrollToLast();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.butterScrollToFirst = function (/* int */ offset) {
        void 0;
        var _offset = offset === undefined ? $(this).first().offset().top : $(this).first().offset().top + offset;
        $('html, body').animate({scrollTop: _offset + 'px'}, 'fast');
    };

    $.fn.butterScrollToLast = function (/* int */ offset) {
        void 0;
        var _offset = offset === undefined ? $(this).last().offset().top : $(this).last().offset().top + offset;
        $('html, body').animate({scrollTop: _offset + 'px'}, 'fast');
    };
}(jQuery));
/**
 * jQuery-Plugin to handle selection style classes on JSF-Component "b:table".
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#someTreeSelector").selectRow( {rowIndex: '6'} );
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.toggleColumnVisibilty = function (renderIds, disableRenderIds) {

        return this.each(function () {
            var $toolbar = $(this);

            var json = JSON.stringify(createColumnVisibilty($toolbar));
            ButterFaces.Ajax.sendRequest($toolbar.attr('id'), 'toggle', renderIds, json, disableRenderIds);
        });

        function createColumnVisibilty($toolbar) {
            var columns = [];

            $toolbar.find('.butter-table-toolbar-column-option input[type=checkbox]').each(function (index, checkbox) {
                var $checkbox = $(checkbox).parent('.butter-table-toolbar-column-option');
                columns.push({
                    identifier: $checkbox.attr('data-column-model-identifier'),
                    visible: $checkbox.find('input[type=checkbox]').is(':checked')
                });
            });

            return columns;
        }
    };

    $.fn.orderColumn = function (renderIds, disableRenderIds, toLeft, columnNumber) {

        return this.each(function () {
            var $toolbar = $(this);

            if (toLeft) {
                //console.log('order column ' + columnNumber + ' to left');
                orderColumnLeft($toolbar, columnNumber);
            } else {
                //console.log('order column ' + columnNumber + ' to right');
                orderColumnRight($toolbar, columnNumber);
            }

            var json = JSON.stringify(createColumnOrder($toolbar));
            ButterFaces.Ajax.sendRequest($toolbar.attr('id'), 'order', renderIds, json, disableRenderIds);
        });

        function createColumnOrder($toolbar) {
            var columns = [];

            $toolbar.find('.butter-table-toolbar-column-option input[type=checkbox]').each(function (index, checkbox) {
                var $checkbox = $(checkbox).parent('.butter-table-toolbar-column-option');
                columns.push({
                    identifier: $checkbox.attr('data-column-model-identifier'),
                    position: index
                });
            });

            return columns;
        }

        function orderColumnLeft(/* jquery toolbar */ $toolbar, columnNumber) {
            //console.log($toolbar);

            var $column = $toolbar.find('li[data-original-column="' + columnNumber + '"]');
            var $nextColumn = $column.prev();

            //console.log($column);
            //console.log($nextColumn);

            var $detachtedColumn = $column.detach();
            $nextColumn.before($detachtedColumn);
        }

        function orderColumnRight(/* jquery toolbar */ $toolbar, columnNumber) {
            //console.log($toolbar);

            var $column = $toolbar.find('li[data-original-column="' + columnNumber + '"]');
            var $nextColumn = $column.next();

            //console.log($column);
            //console.log($nextColumn);

            var $detachtedColumn = $column.detach();
            $nextColumn.after($detachtedColumn);
        }
    };
}(jQuery));
(function ($) {

    $.fn._closePopoverOnBlur = function (/* object */ data) {
        return this.each(function () {
            var root = $(this);

            root.find('.butter-input-component').on('blur', function () {
                root.popover('hide')
            });
        });
    };

    $.fn._butterTooltip = function (/* object */ data) {
        var root = $(this);

        //console.log(data);

        var content = jQuery('[name=' + data.contentByName + ']');

        var newData = {
            trigger: data.trigger,
            title: data.title,
            placement: data.placement,
            placementFunction: data.placementFunction,
            minVerticalOffset: data.minVerticalOffset,
            minHorizontalOffset: data.minHorizontalOffset,
            content: content.html().trim()
        };

        //console.log(newData);

        content.remove();

        root.butterTooltip(newData);
    };

    $.fn.butterTooltip = function (/* object */ data) {
        return this.each(function () {
            var root = $(this);
            var trigger = data.trigger ? data.trigger : 'hover';

            //console.log(data);
            //console.log('placement: ' + placement);
            //console.log('trigger: ' + trigger);
            //console.log('viewport: ' + data.viewport);

            if (data.minVerticalOffset) {
                root.attr('data-tooltip-min-vertical-offset', data.minVerticalOffset);
            }
            if (data.minHorizontalOffset) {
                root.attr('data-tooltip-min-horizontal-offset', data.minHorizontalOffset);
            }
            if (root.next().hasClass('popover')) {
                root.next().remove();
            }


            root.popover({
                trigger: trigger,
                placement: function (popover, source) {
                    return data.placement ? data.placement : (data.placementFunction ? data.placementFunction : new ButterFaces.Tooltip().calculateTooltipPosition(popover, source));
                },
                title: data.title,
                html: true,
                content: data.content,
                viewport: data.viewport
            });
        });
    };
}(jQuery));
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
