/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - rework file opening/closing, and use of managed resource
    ...
**********************************************************************/
package org.datanucleus.store.xml;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;

import javax.transaction.xa.XAResource;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.datanucleus.OMFContext;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.store.connection.AbstractConnectionFactory;
import org.datanucleus.store.connection.AbstractManagedConnection;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.connection.ManagedConnectionResourceListener;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Implementation of a ConnectionFactory for XML.
 * We use a single Managed Connection per transaction/document
 */
public class ConnectionFactoryImpl extends AbstractConnectionFactory
{
    /**
     * Constructor.
     * @param omfContext The OMF context
     * @param resourceType Type of resource (tx, nontx)
     */
    public ConnectionFactoryImpl(OMFContext omfContext, String resourceType)
    {
        super(omfContext, resourceType);
    }

    /**
     * Obtain a connection from the Factory. The connection will be enlisted within the {@link org.datanucleus.Transaction} 
     * associated to the <code>poolKey</code> if "enlist" is set to true.
     * @param poolKey the pool that is bound the connection during its lifecycle (or null)
     * @param transactionOptions Any options for then creating the connection
     * @return the {@link org.datanucleus.store.connection.ManagedConnection}
     */
    public ManagedConnection createManagedConnection(Object poolKey, Map transactionOptions)
    {
        return new ManagedConnectionImpl(omfContext, transactionOptions);
    }

    public class ManagedConnectionImpl extends AbstractManagedConnection
    {
        OMFContext omf;
        Map options;

        /** The XML File. */
        File file;
        
        public ManagedConnectionImpl(OMFContext omf, Map options)
        {
            this.omf = omf;
            this.options = options;
        }

        public void close()
        {
            if (conn == null)
            {
                return;
            }

            for (int i=0; i<listeners.size(); i++)
            {
                ((ManagedConnectionResourceListener)listeners.get(i)).managedConnectionPreClose();
            }
            try
            {
                //this is a kind of commit, do it somewhere else
                try
                {
                    TransformerFactory tf = TransformerFactory.newInstance();
                    Transformer m = tf.newTransformer();
                    DOMSource source = new DOMSource((Document)conn);
                    FileOutputStream os = new FileOutputStream(file);
                    StreamResult result = new StreamResult(os);
                    m.setOutputProperty(OutputKeys.INDENT, "yes");
                    m.transform(source, result);
                    os.close();
                }
                catch (Exception e)
                {
                    throw new NucleusException(e.getMessage(),e);
                }
            }
            finally
            {
                conn = null;
                for (int i=0; i<listeners.size(); i++)
                {
                    ((ManagedConnectionResourceListener)listeners.get(i)).managedConnectionPostClose();
                }
            }
        }

        public Object getConnection()
        {
            if (conn == null)
            {
                // "xml:file:{filename}"
                String url = omf.getStoreManager().getConnectionURL();
                if (url == null)
                {
                    throw new NucleusException("you haven't specified persistence property 'datanucleus.ConnectionURL' (or alias)");
                }
                if (!url.startsWith("xml"))
                {
                    throw new NucleusException("invalid URL: "+url);
                }

                // Split the URL into filename
                String str = url.substring(4); // Omit the prefix
                if (str.indexOf("file:") != 0)
                {
                    throw new NucleusException("invalid URL: "+url);
                }

                try
                {
                    String filename = str.substring(5);
                    file = new File(filename);
                    if (!file.exists())
                    {
                        file.createNewFile();
                        conn = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
                    }
                    else
                    {
                        try
                        {
                            conn = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(file);
                        }
                        catch (SAXException ex)
                        {
                            conn = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
                        }
                    }
                }
                catch (IOException e)
                {
                    throw new NucleusException(e.getMessage(), e);
                }
                catch (ParserConfigurationException e)
                {
                    throw new NucleusException(e.getMessage(), e);
                }
            }
            return conn;
        }

        public XAResource getXAResource()
        {
            return null;
        }
    }
}