/* Copyright (c) 2001 - 2007 TOPP - www.openplans.org. All rights reserved.
 * This code is licensed under the GPL 2.0 license, availible at the root
 * application directory.
 */
package org.vfny.geoserver.wms.responses.map.georss;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;

import org.geoserver.wms.util.WMSRequests;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.map.MapLayer;
import org.geotools.xml.transform.Translator;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.vfny.geoserver.util.Requests;
import org.vfny.geoserver.wms.WMSMapContext;
import org.xml.sax.ContentHandler;
import org.xml.sax.helpers.AttributesImpl;


/**
 * Encodes an RSS feed tagged with geo information.
 *
 * @author Justin Deoliveira, The Open Planning Project, jdeolive@openplans.org
 *
 */
public class RSSGeoRSSTransformer extends GeoRSSTransformerBase {
    public Translator createTranslator(ContentHandler handler) {
        return new RSSGeoRSSTranslator(handler);
    }

    class RSSGeoRSSTranslator extends GeoRSSTranslatorSupport {
        public RSSGeoRSSTranslator(ContentHandler contentHandler) {
            super(contentHandler, null, null);
            nsSupport.declarePrefix("georss", "http://www.georss.org/georss");
            nsSupport.declarePrefix("atom", "http://www.w3.org/2005/Atom");
        }

        public void encode(Object o) throws IllegalArgumentException {
            WMSMapContext map = (WMSMapContext) o;

            AttributesImpl atts = new AttributesImpl();
            atts.addAttribute(null, "version", "version", null, "2.0");

            start("rss", atts);
            start("channel");
            
            element( "title", AtomUtils.getFeedTitle(map) );
            element("description", "Feed auto-generated by GeoServer");
            
            start( "link" );
            cdata(AtomUtils.getFeedURL(map));
            end( "link" );
            
            atts = new AttributesImpl();
            atts.addAttribute(null, "href", "href", null, AtomUtils.getFeedURL(map));
            atts.addAttribute(null, "rel", "rel", null, "self");
            element("atom:link", null, atts);

            //items
            try {
                encodeItems(map);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }

            end("channel");
            end("rss");
        }

        void encodeItems(WMSMapContext map) throws IOException {
            List featureCollections = loadFeatureCollections(map);
            for (Iterator f = featureCollections.iterator(); f.hasNext(); ) {
                FeatureCollection<SimpleFeatureType, SimpleFeature> features = (FeatureCollection) f.next();
                FeatureIterator <SimpleFeature> iterator = null;

                try {
                    iterator = features.features();

                    while (iterator.hasNext()) {
                        SimpleFeature feature = iterator.next();
                        try {
                            encodeItem(feature, map);    
                        }
                        catch( Exception e ) {
                            LOGGER.warning("Encoding failed for feature: " + feature.getID());
                            LOGGER.log(Level.FINE, "", e );
                        }
                        
                    }
                } finally {
                    if (iterator != null) {
                        features.close(iterator);
                    }
                }
                
            }
        }

        void encodeItem(SimpleFeature feature, WMSMapContext map)
            throws IOException {
            start("item");

            try {
                element("title", AtomUtils.getFeatureTitle(feature));
            }
            catch( Exception e ) {
                String msg = "Error occured executing title template for: " + feature.getID();
                LOGGER.log( Level.WARNING, msg, e );
            }
            
            //create the link as getFeature request with fid filter
            start("link");
            cdata(AtomUtils.getEntryURL(feature, map));
            end("link");

            start("guid");
            cdata(AtomUtils.getEntryURL(feature, map));
            end("guid");

            try {
                start("description");
                cdata(AtomUtils.getFeatureDescription(feature));
                end("description");
            }
            catch( Exception e ) {
                String msg = "Error occured executing description template for: " + feature.getID();
                LOGGER.log( Level.WARNING, msg, e );
            }
            
            encodeGeometry(feature);

            end("item");
        }
    }
}
