/*
 * Decompiled with CFR 0.152.
 */
package org.mockserver.socket.tls.jdk;

import com.google.common.net.InetAddresses;
import com.google.common.net.InternetDomainName;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.mockserver.character.Character;
import org.mockserver.file.FileReader;
import org.mockserver.log.model.LogEntry;
import org.mockserver.logging.MockServerLogger;
import org.mockserver.socket.tls.jdk.CertificateSigningRequest;
import org.mockserver.socket.tls.jdk.X509AndPrivateKey;
import org.slf4j.event.Level;
import sun.security.x509.AlgorithmId;
import sun.security.x509.AuthorityKeyIdentifierExtension;
import sun.security.x509.BasicConstraintsExtension;
import sun.security.x509.CertificateAlgorithmId;
import sun.security.x509.CertificateExtensions;
import sun.security.x509.CertificateSerialNumber;
import sun.security.x509.CertificateValidity;
import sun.security.x509.CertificateVersion;
import sun.security.x509.CertificateX509Key;
import sun.security.x509.DNSName;
import sun.security.x509.GeneralName;
import sun.security.x509.GeneralNames;
import sun.security.x509.IPAddressName;
import sun.security.x509.KeyIdentifier;
import sun.security.x509.KeyUsageExtension;
import sun.security.x509.SubjectAlternativeNameExtension;
import sun.security.x509.SubjectKeyIdentifierExtension;
import sun.security.x509.X500Name;
import sun.security.x509.X509CertImpl;
import sun.security.x509.X509CertInfo;

public class X509Generator {
    private static final String BEGIN_CERTIFICATE = "-----BEGIN CERTIFICATE-----";
    private static final String END_CERTIFICATE = "-----END CERTIFICATE-----";
    private static final String BEGIN_PRIVATE_KEY = "-----BEGIN PRIVATE KEY-----";
    private static final String BEGIN_RSA_PRIVATE_KEY = "-----BEGIN RSA PRIVATE KEY-----";
    private static final String END_PRIVATE_KEY = "-----END PRIVATE KEY-----";
    private static final String END_RSA_PRIVATE_KEY = "-----END RSA PRIVATE KEY-----";
    private final MockServerLogger mockServerLogger;

    public X509Generator(MockServerLogger mockServerLogger) {
        this.mockServerLogger = mockServerLogger;
    }

    public X509AndPrivateKey generateRootX509AndPrivateKey(CertificateSigningRequest csr) throws IOException, NoSuchAlgorithmException, CertificateException, InvalidKeyException, NoSuchProviderException, SignatureException {
        KeyPair keyPair = this.generateKeyPair(csr.getKeyPairAlgorithm(), csr.getKeyPairSize());
        X500Name subjectAndIssuer = new X500Name(CertificateSigningRequest.buildDistinguishedName(csr.getCommonName()));
        X509CertInfo x509CertInfo = this.buildX509CertInfo(subjectAndIssuer, subjectAndIssuer, keyPair.getPublic(), csr);
        this.updateWithRootCertificateExtensions(x509CertInfo, keyPair.getPublic());
        return this.signX509KeyPair(keyPair.getPrivate(), keyPair, x509CertInfo, csr.getSigningAlgorithm());
    }

    public X509AndPrivateKey generateLeafX509AndPrivateKey(CertificateSigningRequest csr, String issuerDistinguishingName, String caPrivateKey, X509Certificate caCertificate) throws IOException, NoSuchAlgorithmException, CertificateException, InvalidKeyException, NoSuchProviderException, SignatureException, InvalidKeySpecException {
        PrivateKey privateKey = KeyFactory.getInstance(csr.getKeyPairAlgorithm()).generatePrivate(X509Generator.keySpecFromPEM(caPrivateKey));
        KeyPair keyPair = this.generateKeyPair(csr.getKeyPairAlgorithm(), csr.getKeyPairSize());
        X500Name subject = new X500Name(CertificateSigningRequest.buildDistinguishedName(csr.getCommonName()));
        X500Name issuer = new X500Name(issuerDistinguishingName);
        X509CertInfo x509CertInfo = this.buildX509CertInfo(subject, issuer, keyPair.getPublic(), csr);
        this.updateWithCertificateExtensions(x509CertInfo, keyPair.getPublic(), caCertificate.getPublicKey(), csr.getSubjectAlternativeNames());
        return this.signX509KeyPair(privateKey, keyPair, x509CertInfo, csr.getSigningAlgorithm());
    }

    private KeyPair generateKeyPair(String algorithm, int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator kpg = KeyPairGenerator.getInstance(algorithm);
        kpg.initialize(keySize);
        return kpg.genKeyPair();
    }

    private X509CertInfo buildX509CertInfo(X500Name subject, X500Name issuer, PublicKey publicKey, CertificateSigningRequest csr) throws IOException, NoSuchAlgorithmException, CertificateException {
        X509CertInfo x509CertInfo = new X509CertInfo();
        CertificateValidity interval = new CertificateValidity(CertificateSigningRequest.NOT_BEFORE, CertificateSigningRequest.NOT_AFTER);
        BigInteger sn = new BigInteger(64, new Random());
        x509CertInfo.set("validity", interval);
        x509CertInfo.set("serialNumber", new CertificateSerialNumber(sn));
        x509CertInfo.set("subject", subject);
        x509CertInfo.set("issuer", issuer);
        x509CertInfo.set("key", new CertificateX509Key(publicKey));
        x509CertInfo.set("version", new CertificateVersion(2));
        AlgorithmId algo = new AlgorithmId(AlgorithmId.get(csr.getSigningAlgorithm()).getOID());
        x509CertInfo.set("algorithmID", new CertificateAlgorithmId(algo));
        return x509CertInfo;
    }

    private void updateWithCertificateExtensions(X509CertInfo x509CertInfo, PublicKey publicKey, PublicKey caPublicKey, List<String> subjectAlternativeNames) throws IOException, CertificateException {
        CertificateExtensions certificateExtensions = new CertificateExtensions();
        GeneralNames generalNames = subjectAlternativeNames.stream().filter(StringUtils::isNotBlank).map(this::buildGeneralName).filter(Objects::nonNull).collect(Collector.of(GeneralNames::new, GeneralNames::add, (generalNames1, generalNames2) -> null, new Collector.Characteristics[0]));
        if (!generalNames.isEmpty()) {
            certificateExtensions.set("SubjectAlternativeName", new SubjectAlternativeNameExtension(Boolean.FALSE, generalNames));
        }
        certificateExtensions.set("SubjectKeyIdentifier", new SubjectKeyIdentifierExtension(new KeyIdentifier(publicKey).getIdentifier()));
        certificateExtensions.set("AuthorityKeyIdentifier", new AuthorityKeyIdentifierExtension(new KeyIdentifier(caPublicKey), null, null));
        x509CertInfo.set("extensions", certificateExtensions);
    }

    private void updateWithRootCertificateExtensions(X509CertInfo x509CertInfo, PublicKey publicKey) throws IOException, CertificateException {
        CertificateExtensions certificateExtensions = new CertificateExtensions();
        certificateExtensions.set("BasicConstraints", new BasicConstraintsExtension(true, true, -1));
        boolean[] keyUsage = new boolean[9];
        keyUsage[5] = true;
        certificateExtensions.set("KeyUsage", new KeyUsageExtension(keyUsage));
        certificateExtensions.set("SubjectKeyIdentifier", new SubjectKeyIdentifierExtension(new KeyIdentifier(publicKey).getIdentifier()));
        x509CertInfo.set("extensions", certificateExtensions);
    }

    private GeneralName buildGeneralName(String subjectAlternativeName) {
        if (InetAddresses.isUriInetAddress(subjectAlternativeName)) {
            try {
                return new GeneralName(new IPAddressName(subjectAlternativeName));
            }
            catch (Throwable throwable) {
                this.mockServerLogger.logEvent(new LogEntry().setLogLevel(Level.WARN).setMessageFormat("Unable to use ip address with the value \"" + subjectAlternativeName + "\" as Subject Alternative Name (SAN) for X509 as JDK does not support SANs with that format"));
            }
        } else if (InternetDomainName.isValid(subjectAlternativeName)) {
            try {
                return new GeneralName(new DNSName(subjectAlternativeName));
            }
            catch (Throwable throwable) {
                this.mockServerLogger.logEvent(new LogEntry().setLogLevel(Level.WARN).setMessageFormat("Unable to use domain name with the value \"" + subjectAlternativeName + "\" as Subject Alternative Name (SAN) for X509 as JDK does not support SANs with that format"));
            }
        }
        return null;
    }

    private X509AndPrivateKey signX509KeyPair(PrivateKey privateKey, KeyPair keyPair, X509CertInfo x509CertInfo, String signatureAlgorithm) throws CertificateException, NoSuchAlgorithmException, IOException, InvalidKeyException, NoSuchProviderException, SignatureException {
        x509CertInfo.set("algorithmID", new CertificateAlgorithmId(AlgorithmId.get(signatureAlgorithm)));
        X509CertImpl cert = new X509CertImpl(x509CertInfo);
        cert.sign(privateKey, signatureAlgorithm);
        return new X509AndPrivateKey().setPrivateKey(X509Generator.privateKeyToPEM(keyPair.getPrivate().getEncoded())).setCert(X509Generator.certToPEM(new byte[][]{cert.getEncoded()}));
    }

    static String privateKeyToPEM(byte[] privateKey) {
        return X509Generator.toPEM(privateKey, BEGIN_PRIVATE_KEY, END_PRIVATE_KEY);
    }

    public static String certToPEM(X509Certificate ... x509Certificates) throws CertificateEncodingException {
        StringBuilder pem = new StringBuilder();
        for (X509Certificate x509Certificate : x509Certificates) {
            pem.append(X509Generator.toPEM(x509Certificate.getEncoded(), BEGIN_CERTIFICATE, END_CERTIFICATE)).append(Character.NEW_LINE);
        }
        return pem.toString();
    }

    public static String certToPEM(byte[] ... x509Certificates) {
        StringBuilder pem = new StringBuilder();
        for (byte[] x509Certificate : x509Certificates) {
            pem.append(X509Generator.toPEM(x509Certificate, BEGIN_CERTIFICATE, END_CERTIFICATE)).append(Character.NEW_LINE);
        }
        return pem.toString();
    }

    private static String toPEM(byte[] key, String begin, String end) {
        Base64.Encoder encoder = Base64.getMimeEncoder(64, System.lineSeparator().getBytes());
        return begin + System.lineSeparator() + encoder.encodeToString(key) + System.lineSeparator() + end;
    }

    static byte[] privateKeyBytesFromPEM(String pem) {
        if (pem.contains(BEGIN_RSA_PRIVATE_KEY) || pem.contains(END_RSA_PRIVATE_KEY)) {
            new MockServerLogger().logEvent(new LogEntry().setType(LogEntry.LogMessageType.EXCEPTION).setLogLevel(Level.ERROR).setMessageFormat("Private key provided in unsupported PKCS#1 only PKCS#8 format is support, to convert use openssl, for example{}").setArguments("openssl pkcs8 -topk8 -inform PEM -in private_key_PKCS_1.pem -out private_key_PKCS_8.pem -nocrypt"));
        }
        return Base64.getMimeDecoder().decode(pem.replaceFirst(BEGIN_PRIVATE_KEY, "").replaceFirst(BEGIN_RSA_PRIVATE_KEY, "").replaceFirst(END_PRIVATE_KEY, "").replaceFirst(END_RSA_PRIVATE_KEY, ""));
    }

    public static KeySpec keySpecFromPEM(String pem) {
        return new PKCS8EncodedKeySpec(X509Generator.privateKeyBytesFromPEM(pem));
    }

    public static RSAPrivateKey privateKeyFromPEMFile(String filename) {
        try {
            return X509Generator.privateKeyFromPEM(FileReader.readFileFromClassPathOrPath(filename));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading private key from PEM file", e);
        }
    }

    public static RSAPrivateKey privateKeyFromPEM(String pem) {
        try {
            return (RSAPrivateKey)KeyFactory.getInstance("RSA").generatePrivate(X509Generator.keySpecFromPEM(pem));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading private key from PEM file", e);
        }
    }

    public static X509Certificate x509FromPEMFile(String filename) {
        try {
            return X509Generator.x509FromPEM(FileReader.openStreamToFileFromClassPathOrPath(filename));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading X509 from PEM file " + filename, e);
        }
    }

    public static X509Certificate x509FromPEM(String pem) {
        try {
            return X509Generator.x509FromPEM(new ByteArrayInputStream(pem.getBytes()));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading X509 from PEM " + Character.NEW_LINE + pem, e);
        }
    }

    private static X509Certificate x509FromPEM(InputStream inputStream) {
        try {
            return (X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(inputStream);
        }
        catch (Exception e) {
            throw new RuntimeException("Exception creating X509 from PEM", e);
        }
    }

    public static List<X509Certificate> x509ChainFromPEMFile(String filename) {
        try {
            return X509Generator.x509ChainFromPEM(FileReader.openStreamToFileFromClassPathOrPath(filename));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading X509 from PEM file " + filename, e);
        }
    }

    public static List<X509Certificate> x509ChainFromPEM(String pem) {
        try {
            return X509Generator.x509ChainFromPEM(new ByteArrayInputStream(pem.getBytes()));
        }
        catch (Exception e) {
            throw new RuntimeException("Exception reading X509 from PEM " + Character.NEW_LINE + pem, e);
        }
    }

    private static List<X509Certificate> x509ChainFromPEM(InputStream inputStream) {
        try {
            return CertificateFactory.getInstance("X.509").generateCertificates(inputStream).stream().filter(certificate -> certificate instanceof X509Certificate).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new RuntimeException("Exception creating X509 from PEM", e);
        }
    }

    public static boolean validX509PEMFileExists(String filename) {
        try {
            return X509Generator.x509FromPEMFile(filename) != null;
        }
        catch (Exception e) {
            return false;
        }
    }
}

