/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.backup;

import ch.qos.logback.classic.LoggerContext;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.NoSuchElementException;
import org.neo4j.backup.BackupExtensionService;
import org.neo4j.backup.BackupServer;
import org.neo4j.backup.BackupService;
import org.neo4j.com.ComException;
import org.neo4j.consistency.ConsistencyCheckSettings;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.Args;
import org.neo4j.helpers.HostnamePort;
import org.neo4j.helpers.Service;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.kernel.DefaultFileSystemAbstraction;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.nioneo.store.FileSystemAbstraction;
import org.neo4j.kernel.impl.nioneo.store.MismatchingStoreIdException;
import org.neo4j.kernel.impl.storemigration.LogFiles;
import org.neo4j.kernel.impl.storemigration.StoreFileType;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedByConfigurationException;
import org.neo4j.kernel.impl.storemigration.legacystore.v20.StoreFile20;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.logging.LogbackService;
import org.neo4j.kernel.logging.Logging;
import org.neo4j.kernel.logging.SystemOutLogging;
import org.slf4j.impl.StaticLoggerBinder;

public class BackupTool {
    private static final String TO = "to";
    private static final String HOST = "host";
    private static final String PORT = "port";
    @Deprecated
    private static final String FROM = "from";
    private static final String VERIFY = "verify";
    private static final String CONFIG = "config";
    private static final String FORENSICS = "gather-forensics";
    public static final String DEFAULT_SCHEME = "single";
    static final String MISMATCHED_STORE_ID = "You tried to perform a backup from database %s, but the target directory contained a backup from database %s. ";
    static final String WRONG_FROM_ADDRESS_SYNTAX = "Please properly specify a location to backup in the form " + BackupTool.dash("host") + " <host> " + BackupTool.dash("port") + " <port>";
    static final String UNKNOWN_SCHEMA_MESSAGE_PATTERN = "%s was specified as a backup module but it was not found. Please make sure that the implementing service is on the classpath.";
    static final String NO_SOURCE_SPECIFIED = "Please specify " + BackupTool.dash("host") + " and optionally " + BackupTool.dash("port") + ", examples:\n" + "  " + BackupTool.dash("host") + " 192.168.1.34\n" + "  " + BackupTool.dash("host") + " 192.168.1.34 " + BackupTool.dash("port") + " 1234";
    private final BackupService backupService;
    private final PrintStream systemOut;
    private final FileSystemAbstraction fs;

    public static void main(String[] args) {
        BackupTool tool = new BackupTool(new BackupService((FileSystemAbstraction)new DefaultFileSystemAbstraction()), System.out);
        try {
            BackupService.BackupOutcome backupOutcome = tool.run(args);
            if (!backupOutcome.isConsistent()) {
                BackupTool.exitFailure("WARNING: The database is inconsistent.");
            }
        }
        catch (ToolFailureException e) {
            if (e.getCause() != null) {
                e.getCause().printStackTrace(System.out);
            }
            BackupTool.exitFailure(e.getMessage());
        }
    }

    BackupTool(BackupService backupService, PrintStream systemOut) {
        this.backupService = backupService;
        this.systemOut = systemOut;
        this.fs = new DefaultFileSystemAbstraction();
    }

    BackupService.BackupOutcome run(String[] args) throws ToolFailureException {
        Args arguments = new Args(args);
        if (!arguments.hasNonNull(TO)) {
            throw new ToolFailureException("Specify target location with " + BackupTool.dash(TO) + " <target-directory>");
        }
        if (arguments.hasNonNull(FROM) && !arguments.has(HOST) && !arguments.has(PORT)) {
            return this.runBackupWithLegacyArgs(arguments);
        }
        if (arguments.hasNonNull(HOST)) {
            return this.runBackup(arguments);
        }
        throw new ToolFailureException(NO_SOURCE_SPECIFIED);
    }

    private BackupService.BackupOutcome runBackupWithLegacyArgs(Args args) throws ToolFailureException {
        String from = args.get(FROM).trim();
        String to = args.get(TO).trim();
        boolean verify = args.getBoolean(VERIFY, Boolean.valueOf(true), Boolean.valueOf(true));
        Config tuningConfiguration = BackupTool.readTuningConfiguration(TO, args);
        boolean forensics = args.getBoolean(FORENSICS, Boolean.valueOf(false), Boolean.valueOf(true));
        URI backupURI = BackupTool.resolveBackupUri(from, args, tuningConfiguration);
        HostnamePort hostnamePort = BackupTool.newHostnamePort(backupURI);
        return this.executeBackup(hostnamePort, to, verify, tuningConfiguration, forensics);
    }

    private BackupService.BackupOutcome runBackup(Args args) throws ToolFailureException {
        String host = args.get(HOST).trim();
        int port = args.getNumber(PORT, (Number)BackupServer.DEFAULT_PORT).intValue();
        String to = args.get(TO).trim();
        boolean verify = args.getBoolean(VERIFY, Boolean.valueOf(true), Boolean.valueOf(true));
        Config tuningConfiguration = BackupTool.readTuningConfiguration(TO, args);
        boolean forensics = args.getBoolean(FORENSICS, Boolean.valueOf(false), Boolean.valueOf(true));
        if (host.contains(":")) {
            if (!host.startsWith("[")) {
                host = "[" + host;
            }
            if (!host.endsWith("]")) {
                host = host + "]";
            }
        }
        URI backupURI = BackupTool.newURI("single://" + host + ":" + port);
        HostnamePort hostnamePort = BackupTool.newHostnamePort(backupURI);
        return this.executeBackup(hostnamePort, to, verify, tuningConfiguration, forensics);
    }

    private BackupService.BackupOutcome executeBackup(HostnamePort hostnamePort, String to, boolean verify, Config tuningConfiguration, boolean forensics) throws ToolFailureException {
        try {
            this.systemOut.println("Performing backup from '" + hostnamePort + "'");
            return this.doBackup(hostnamePort, to, verify, tuningConfiguration, forensics);
        }
        catch (TransactionFailureException tfe) {
            if (tfe.getCause() instanceof UpgradeNotAllowedByConfigurationException) {
                try {
                    this.systemOut.println("The database present in the target directory is of an older version. Backing that up in target and performing a full backup from source");
                    BackupTool.moveExistingDatabase(this.fs, to);
                }
                catch (IOException e) {
                    throw new ToolFailureException("There was a problem moving the old database out of the way - cannot continue, aborting.", e);
                }
                return this.doBackup(hostnamePort, to, verify, tuningConfiguration, forensics);
            }
            throw new ToolFailureException("TransactionFailureException from existing backup at '" + hostnamePort + "'.", tfe);
        }
    }

    private BackupService.BackupOutcome doBackup(HostnamePort hostnamePort, String to, boolean checkConsistency, Config config, boolean forensics) throws ToolFailureException {
        try {
            String host = hostnamePort.getHost();
            int port = hostnamePort.getPort();
            BackupService.BackupOutcome outcome = this.backupService.doIncrementalBackupOrFallbackToFull(host, port, to, checkConsistency, config, forensics);
            this.systemOut.println("Done");
            return outcome;
        }
        catch (MismatchingStoreIdException e) {
            this.systemOut.println("Backup failed.");
            throw new ToolFailureException(String.format(MISMATCHED_STORE_ID, e.getExpected(), e.getEncountered()));
        }
        catch (ComException e) {
            throw new ToolFailureException("Couldn't connect to '" + hostnamePort + "'", e);
        }
    }

    private static Config readTuningConfiguration(String storeDir, Args arguments) throws ToolFailureException {
        Map specifiedProperties = MapUtil.stringMap((String[])new String[0]);
        String propertyFilePath = arguments.get(CONFIG, null);
        if (propertyFilePath != null) {
            File propertyFile = new File(propertyFilePath);
            try {
                specifiedProperties = MapUtil.load((File)propertyFile);
            }
            catch (IOException e) {
                throw new ToolFailureException(String.format("Could not read configuration properties file [%s]", propertyFilePath), e);
            }
        }
        specifiedProperties.put(GraphDatabaseSettings.store_dir.name(), storeDir);
        return new Config(specifiedProperties, new Class[]{GraphDatabaseSettings.class, ConsistencyCheckSettings.class});
    }

    private static URI resolveBackupUri(String from, Args arguments, Config config) throws ToolFailureException {
        if (from.contains(",")) {
            if (!from.startsWith("ha://")) {
                BackupTool.checkNoSchemaIsPresent(from);
                from = "ha://" + from;
            }
            return BackupTool.resolveUriWithProvider("ha", from, arguments, config);
        }
        if (!from.startsWith("single://")) {
            from = from.replace("ha://", "");
            BackupTool.checkNoSchemaIsPresent(from);
            from = "single://" + from;
        }
        return BackupTool.newURI(from);
    }

    private static void checkNoSchemaIsPresent(String address) throws ToolFailureException {
        if (address.contains("://")) {
            throw new ToolFailureException(WRONG_FROM_ADDRESS_SYNTAX);
        }
    }

    private static URI newURI(String uriString) throws ToolFailureException {
        try {
            return new URI(uriString);
        }
        catch (URISyntaxException e) {
            throw new ToolFailureException(WRONG_FROM_ADDRESS_SYNTAX);
        }
    }

    private static URI resolveUriWithProvider(String providerName, String from, Args args, Config config) throws ToolFailureException {
        BackupExtensionService service;
        try {
            service = (BackupExtensionService)Service.load(BackupExtensionService.class, (String)providerName);
        }
        catch (NoSuchElementException e) {
            throw new ToolFailureException(String.format(UNKNOWN_SCHEMA_MESSAGE_PATTERN, providerName));
        }
        try {
            return service.resolve(from, args, BackupTool.newLogging(config));
        }
        catch (Throwable t) {
            throw new ToolFailureException(t.getMessage());
        }
    }

    private static HostnamePort newHostnamePort(URI backupURI) throws ToolFailureException {
        if (backupURI == null || backupURI.getHost() == null) {
            throw new ToolFailureException(WRONG_FROM_ADDRESS_SYNTAX);
        }
        String host = backupURI.getHost();
        int port = backupURI.getPort();
        if (port == -1) {
            port = BackupServer.DEFAULT_PORT;
        }
        return new HostnamePort(host, port);
    }

    private static Logging newLogging(Config config) {
        SystemOutLogging logging;
        try {
            BackupTool.class.getClassLoader().loadClass("ch.qos.logback.classic.LoggerContext");
            LifeSupport life = new LifeSupport();
            LogbackService logbackService = (LogbackService)life.add((Object)new LogbackService(config, (LoggerContext)StaticLoggerBinder.getSingleton().getLoggerFactory(), "neo4j-backup-logback.xml"));
            life.start();
            logging = logbackService;
        }
        catch (Throwable e) {
            logging = new SystemOutLogging();
        }
        return logging;
    }

    private static void moveExistingDatabase(FileSystemAbstraction fs, String to) throws IOException {
        File toDir = new File(to);
        File backupDir = new File(toDir, "old-version");
        if (!fs.mkdir(backupDir)) {
            throw new IOException("Trouble making target backup directory " + backupDir.getAbsolutePath());
        }
        StoreFile20.move((FileSystemAbstraction)fs, (File)toDir, (File)backupDir, (Iterable)StoreFile20.legacyStoreFiles(), (boolean)false, (boolean)false, (StoreFileType[])StoreFileType.values());
        LogFiles.move((FileSystemAbstraction)fs, (File)toDir, (File)backupDir);
    }

    private static String dash(String name) {
        return "-" + name;
    }

    static void exitFailure(String msg) {
        System.out.println(msg);
        System.exit(1);
    }

    static class ToolFailureException
    extends Exception {
        ToolFailureException(String message) {
            super(message);
        }

        ToolFailureException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

