/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ogm.persistence.examples.drink;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import org.assertj.core.api.Assertions;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.neo4j.ogm.domain.drink.Beverage;
import org.neo4j.ogm.domain.drink.Ingredient;
import org.neo4j.ogm.domain.drink.Manufacturer;
import org.neo4j.ogm.domain.drink.Owns;
import org.neo4j.ogm.session.Session;
import org.neo4j.ogm.session.SessionFactory;
import org.neo4j.ogm.session.event.EventListener;
import org.neo4j.ogm.session.event.EventListenerAdapter;
import org.neo4j.ogm.testutil.MultiDriverTestClass;

public class DrinkIntegrationTest
extends MultiDriverTestClass {
    private static SessionFactory sf;
    private Session session;

    @BeforeClass
    public static void setUpClass() throws Exception {
        sf = new SessionFactory(driver, new String[]{Beverage.class.getPackage().getName()});
        sf.register((EventListener)new EventListenerAdapter());
    }

    @Before
    public void setUp() throws Exception {
        this.session = sf.openSession();
        this.session.purgeDatabase();
    }

    @Test
    public void shouldSaveAndLoadEntityWithoutGraphId() throws Exception {
        Beverage beverage = new Beverage("Pilsner Urquell");
        this.session.save((Object)beverage);
        this.assertBeverageInDB(beverage);
        this.session.clear();
        this.assertBeverageInDB(beverage);
    }

    @Test
    public void shouldLoadAllBeverages() throws Exception {
        Beverage pilsner = new Beverage("Pilsner Urquell");
        this.session.save((Object)pilsner);
        Beverage budweiser = new Beverage("Budweiser Budvar");
        this.session.save((Object)budweiser);
        Collection beverages = this.session.loadAll(Beverage.class);
        Assertions.assertThat((Iterable)beverages).containsOnly((Object[])new Beverage[]{pilsner, budweiser});
        this.session.clear();
        Collection loaded = this.session.loadAll(Beverage.class);
        Assertions.assertThat((Iterable)loaded).extracting(Beverage::getUuid).containsOnly((Object[])new String[]{pilsner.getUuid(), budweiser.getUuid()});
    }

    @Test
    public void shouldLoadAllByInstances() throws Exception {
        Beverage pilsner = new Beverage("Pilsner Urquell");
        this.session.save((Object)pilsner);
        Collection beverages = this.session.loadAll(Arrays.asList(pilsner));
        Assertions.assertThat((Iterable)beverages).containsOnly((Object[])new Beverage[]{pilsner});
    }

    @Test
    public void loadReturnsSameInstance() throws Exception {
        Beverage beverage = new Beverage("Pilsner Urquell");
        this.session.save((Object)beverage);
        Beverage loaded = (Beverage)this.session.load(Beverage.class, (Serializable)((Object)beverage.getUuid()));
        Assertions.assertThat((Object)loaded).isSameAs((Object)beverage);
    }

    @Test
    public void shouldCorrectlyUpdateInDb() throws Exception {
        Beverage beverage = new Beverage("Pilsenerer Urquell");
        this.session.save((Object)beverage);
        beverage.setName("Pilsner Urquell");
        this.session.save((Object)beverage);
        this.assertBeverageInDB(beverage);
        this.session.clear();
        this.assertBeverageInDB(beverage);
    }

    @Test
    public void shouldDeleteEntity() throws Exception {
        Beverage beverage = new Beverage("Pilsenerer Urquell");
        this.session.save((Object)beverage);
        this.session.delete((Object)beverage);
        this.assertNoBeverage();
    }

    @Test
    public void shouldDeleteRelationship() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Beverage pilsner = new Beverage("Pilsenerer Urquell");
        prazdroj.addBeverage(pilsner);
        this.session.save((Object)prazdroj);
        prazdroj.getBeverages().clear();
        pilsner.setManufacturer(null);
        this.session.save((Object)prazdroj);
        Manufacturer loadedPrazdroj = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)prazdroj.getUuid()));
        Assertions.assertThat(loadedPrazdroj.getBeverages()).isNullOrEmpty();
        this.session.clear();
        loadedPrazdroj = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)prazdroj.getUuid()));
        Assertions.assertThat(loadedPrazdroj.getBeverages()).isNullOrEmpty();
    }

    @Test
    public void shouldDeleteRelationshipEntity() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        asahi.acquired(2017, prazdroj);
        this.session.save((Object)asahi);
        asahi.getAcquisitions().clear();
        this.session.save((Object)asahi);
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Assertions.assertThat(loaded.getAcquisitions()).isNullOrEmpty();
        this.session.clear();
        loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Assertions.assertThat(loaded.getAcquisitions()).isNullOrEmpty();
    }

    private void assertBeverageInDB(Beverage beverage) {
        Beverage loaded = (Beverage)this.session.load(Beverage.class, (Serializable)((Object)beverage.getUuid()));
        Assertions.assertThat((String)loaded.getUuid()).isEqualTo((Object)beverage.getUuid());
        Assertions.assertThat((String)loaded.getName()).isEqualTo((Object)beverage.getName());
    }

    @Test
    public void shouldSaveEntityWithoutAndWithGraphID() throws Exception {
        Beverage pilsner = new Beverage("Pilsner Urquell");
        Ingredient water = new Ingredient("Water");
        pilsner.addIngredient(water);
        this.session.save((Object)pilsner);
        Assertions.assertThat((String)pilsner.getUuid()).isNotNull();
        Assertions.assertThat((Long)water.getId()).isNotNull();
        this.assertBeverageAndIngredientInDB(pilsner, water);
        this.session.clear();
        this.assertBeverageAndIngredientInDB(pilsner, water);
    }

    @Test
    public void shouldSaveEntitydWithGraphIdAndWIthoutId() throws Exception {
        Beverage pilsner = new Beverage("Pilsner Urquell");
        Ingredient water = new Ingredient("Water");
        pilsner.addIngredient(water);
        this.session.save((Object)water);
        Assertions.assertThat((Long)water.getId()).isNotNull();
        Assertions.assertThat((String)pilsner.getUuid()).isNotNull();
        this.assertBeverageAndIngredientInDB(pilsner, water);
        this.session.clear();
        this.assertBeverageAndIngredientInDB(pilsner, water);
    }

    @Test
    public void shouldSaveEntitiesWithoutGraphIds() throws Exception {
        Beverage pilsner = new Beverage("Pilsner Urquell");
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        prazdroj.addBeverage(pilsner);
        this.session.save((Object)pilsner);
        Beverage loaded = (Beverage)this.session.load(Beverage.class, (Serializable)((Object)pilsner.getUuid()));
        Assertions.assertThat((Object)loaded.getManufacturer()).isSameAs((Object)prazdroj);
        this.assertBeverageAndManufacturerInDB(pilsner, prazdroj);
        this.session.clear();
        this.assertBeverageAndManufacturerInDB(pilsner, prazdroj);
    }

    @Test
    public void shouldSaveAndLoadRelationshipEntityWithoutId() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        asahi.acquired(2017, prazdroj);
        this.session.save((Object)asahi);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Assertions.assertThat((String)loaded.getName()).isEqualTo((Object)"Asahi Breweries, Ltd.");
        Owns acquisition = loaded.getAcquisitions().iterator().next();
        Assertions.assertThat((String)acquisition.getOwnee().getName()).isEqualTo((Object)"Plze\u0148sk\u00fd Prazdroj, a. s.");
    }

    @Test
    public void shouldSaveAndLoadRelationshipEntityWithoutIdDirect() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        asahi.acquired(2017, prazdroj);
        Owns owns = asahi.getAcquisitions().iterator().next();
        this.session.save((Object)owns);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Assertions.assertThat((String)loaded.getName()).isEqualTo((Object)"Asahi Breweries, Ltd.");
        Owns acquisition = loaded.getAcquisitions().iterator().next();
        Assertions.assertThat((String)acquisition.getOwnee().getName()).isEqualTo((Object)"Plze\u0148sk\u00fd Prazdroj, a. s.");
    }

    @Test
    public void shouldUpdateRelationshipEntityTransitive() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        asahi.acquired(1917, prazdroj);
        this.session.save((Object)asahi);
        Owns owns = asahi.getAcquisitions().iterator().next();
        owns.setAcquiredYear(2017);
        this.session.save((Object)asahi);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Owns acquisition = loaded.getAcquisitions().iterator().next();
        Assertions.assertThat((int)acquisition.getAcquiredYear()).isEqualTo(2017);
    }

    @Test
    public void shouldUpdateRelationshipEntityDirect() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        asahi.acquired(1917, prazdroj);
        Owns owns = asahi.getAcquisitions().iterator().next();
        this.session.save((Object)owns);
        owns.setAcquiredYear(2017);
        this.session.save((Object)owns);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Owns acquisition = loaded.getAcquisitions().iterator().next();
        Assertions.assertThat((int)acquisition.getAcquiredYear()).isEqualTo(2017);
    }

    @Test
    public void shouldCreateRelationshipEntityDirect() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        Manufacturer asahi = new Manufacturer("Asahi Breweries, Ltd.");
        this.session.save((Object)prazdroj);
        this.session.save((Object)asahi);
        Owns owns = new Owns(asahi, prazdroj, 2017);
        this.session.save((Object)owns);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)asahi.getUuid()));
        Owns acquisition = loaded.getAcquisitions().iterator().next();
        Assertions.assertThat((int)acquisition.getAcquiredYear()).isEqualTo(2017);
    }

    @Test
    public void shouldAddAndRemoveLabels() throws Exception {
        Manufacturer prazdroj = new Manufacturer("Plze\u0148sk\u00fd Prazdroj, a. s.");
        this.session.save((Object)prazdroj);
        prazdroj.addLabel("BestBeer");
        this.session.save((Object)prazdroj);
        this.session.clear();
        Manufacturer loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)prazdroj.getUuid()));
        Assertions.assertThat(loaded.getLabels()).containsOnly((Object[])new String[]{"BestBeer"});
        prazdroj.setLabels(Collections.emptySet());
        this.session.save((Object)prazdroj);
        loaded = (Manufacturer)this.session.load(Manufacturer.class, (Serializable)((Object)prazdroj.getUuid()));
        Assertions.assertThat(loaded.getLabels()).isEmpty();
    }

    private void assertBeverageAndManufacturerInDB(Beverage beverage, Manufacturer manufacturer) {
        Beverage loaded = (Beverage)this.session.load(Beverage.class, (Serializable)((Object)beverage.getUuid()));
        Assertions.assertThat((String)loaded.getName()).isEqualTo((Object)beverage.getName());
        Assertions.assertThat((String)loaded.getManufacturer().getName()).isEqualTo((Object)manufacturer.getName());
    }

    private void assertBeverageAndIngredientInDB(Beverage pilsner, Ingredient water) {
        Beverage loaded = (Beverage)this.session.load(Beverage.class, (Serializable)((Object)pilsner.getUuid()));
        Assertions.assertThat((String)loaded.getName()).isEqualTo((Object)"Pilsner Urquell");
        Assertions.assertThat(loaded.getIngredients()).extracting(Ingredient::getName).containsOnly((Object[])new String[]{water.getName()});
    }

    private void assertNoBeverage() {
        Collection beverages = this.session.loadAll(Beverage.class);
        Assertions.assertThat((Iterable)beverages).hasSize(0);
    }
}

