/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.common.profile.logic;

import java.util.Collection;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.saml.ext.saml2mdrpi.RegistrationInfo;
import org.opensaml.saml.ext.saml2mdrpi.RegistrationPolicy;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Predicate to determine whether one of a set of names matches an entity's
 * {@link RegistrationPolicy}.
 */
public class RegistrationPolicyPredicate extends AbstractRegistrationInfoPredicate {

    /** Policies to match on. */
    @Nonnull private final Set<String> policySet;
    
    /**
     * Constructor.
     * 
     * @param policies the policies to test for
     */
    public RegistrationPolicyPredicate(@Nullable @ParameterName(name="policies") final Collection<String> policies) {
        policySet = CollectionSupport.copyToSet(StringSupport.normalizeStringCollection(policies));
    }

    /**
     * Get the policy criteria.
     * 
     * @return  the policy criteria
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getPolicies() {
        return policySet;
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doApply(@Nonnull final RegistrationInfo info) {
        for (final RegistrationPolicy policy : info.getRegistrationPolicies()) {
            if (policySet.contains(policy.getURI())) {
                return true;
            }
        }
        
        return false;
    }

}