/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.soap.client.security;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.InOutOperationContext;
import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.messaging.context.navigate.ContextDataLookupFunction;
import org.opensaml.messaging.context.navigate.RecursiveTypedParentContextLookup;

import net.shibboleth.shared.logic.Constraint;

/**
 * Function to resolve SOAP client security profile ID from message context.
 */
public class SOAPClientSecurityProfileIdLookupFunction implements ContextDataLookupFunction<MessageContext, String> {
    
    /** Lookup function for {@link SOAPClientSecurityContext}. */
    @Nonnull private Function<MessageContext, SOAPClientSecurityContext> soapContextLookup;
    
    /**
     * Constructor.
     */
    public SOAPClientSecurityProfileIdLookupFunction() {
        soapContextLookup =
                new ChildContextLookup<>(SOAPClientSecurityContext.class).compose(
                        new RecursiveTypedParentContextLookup<>(InOutOperationContext.class));
    }
    
    /**
     * Set lookup function for for {@link SOAPClientSecurityContext}.
     * 
     * @param lookup the lookup function
     */
    public void setSOAPClientSecurityContextLookup(
            @Nonnull final Function<MessageContext, SOAPClientSecurityContext> lookup) {
       soapContextLookup = Constraint.isNotNull(lookup, "SOAPClientSecurityContext lookup function was null") ;
    }

    /** {@inheritDoc} */
    @Nullable public String apply(@Nullable final MessageContext messageContext) {
        if (messageContext == null) {
            return null;
        }
        
        final SOAPClientSecurityContext context = soapContextLookup.apply(messageContext);
        if (context != null) {
            return context.getSecurityConfigurationProfileId();
        }
        return null;
    }

}