// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.collection;

import java.util.HashMap;
import java.util.Map;

import org.refcodes.collection.Dictionary.MutableDictionary.DictionaryBuilder;
import org.refcodes.structure.Property;

/**
 * The {@link Properties} are a specialization of the {@link Dictionary}. As the
 * name says its on a {@link String} pairs key/value collection as properties
 * usually occur in pure text form to be converted to the required data types.
 * For this reason, the {@link Properties} interface provides additional
 * conversion methods.
 */
public interface Properties extends Dictionary<String, String> {

	/**
	 * This method creates a {@link java.util.Properties} instance from this
	 * {@link Properties} instance's elements (key/value-pairs) as of
	 * interoperability reasons.
	 * 
	 * @return A {@link java.util.Properties} object from the herein contained
	 *         key/value-pairs .
	 */
	default java.util.Properties toProperties() {
		java.util.Properties theProperties = new java.util.Properties();
		for ( String eKey : keySet() ) {
			theProperties.put( eKey, get( eKey ) );
		}
		return theProperties;
	}

	/**
	 * This method creates a {@link Map} instance from this {@link Properties}
	 * instance's elements (key/value-pairs) as of interoperability reasons.
	 * 
	 * @return A {@link Map} object from the herein contained key/value-pairs .
	 */
	default Map<String, String> toMap() {
		Map<String, String> theProperties = new HashMap<>();
		for ( String eKey : keySet() ) {
			theProperties.put( eKey, get( eKey ) );
		}
		return theProperties;
	}

	public interface ClearableProperties extends Properties, ClearableDictionary<String, String> {}

	public interface MutableProperties extends ClearableProperties, MutableDictionary<String, String> {

		/**
		 * Puts the key/value-pair from the provided {@link Property}.
		 * 
		 * @param aProperty The property's key/value to be inserted.
		 * 
		 * @return The property value overwritten by the provided property.
		 */
		String put( Property aProperty );

		/**
		 * This method inserts all elements (key/value-pairs) found in the
		 * provided {@link Map} instances of interoperability reasons.
		 * 
		 * @param aProperties A {@link Map} containing the key/value-pairs to be
		 *        inserted.
		 */
		default void putAll( Map<String, String> aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, get( (String) eKey ) );
			}
		}

		/**
		 * This method inserts all elements (key/value-pairs) found in the
		 * provided {@link Properties} instances of interoperability reasons.
		 * 
		 * @param aProperties A {@link Properties} containing the key/value
		 *        pairs to be inserted.
		 */
		default void putAll( Properties aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, get( (String) eKey ) );
			}
		}

		/**
		 * This method inserts all elements (key/value-pairs) found in the
		 * provided {@link java.util.Properties} instances of interoperability
		 * reasons.
		 * 
		 * @param aProperties A {@link java.util.Properties} containing the
		 *        key/value-pairs to be inserted.
		 */
		default void putAll( java.util.Properties aProperties ) {
			for ( Object eKey : aProperties.keySet() ) {
				put( (String) eKey, get( (String) eKey ) );
			}
		}

	}

	public interface PropertiesBuilder extends DictionaryBuilder<String, String, Property, PropertiesBuilder>, MutableProperties {

		PropertiesBuilder withPut( String aKey, String aValue );

		PropertiesBuilder withPut( Property aProperty );

	}

}
