// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cryptography.tests;

import java.io.File;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;

import org.junit.Ignore;
import org.junit.Test;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.factories.impls.RuntimeLoggerFactorySingleton;

import edu.vt.middleware.crypt.asymmetric.AsymmetricAlgorithm;
import edu.vt.middleware.crypt.asymmetric.PublicKeyUtils;
import edu.vt.middleware.crypt.asymmetric.RSA;
import edu.vt.middleware.crypt.util.Base64Converter;
import edu.vt.middleware.crypt.util.CryptWriter;

public class CryptographyWorkshopTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.getInstance().createInstance();

	private static final String PATH_TO_SERVICE_KEY = "/tmp/rsa-service-key.pem";
	private static final String PATH_TO_SERVICE_PUB_KEY = "/tmp/rsa-service-pub-key.pem";
	private static final String PATH_TO_SERVER_KEY = "/tmp/rsa-server-key.pem";
	private static final String PATH_TO_SERVER_PUB_KEY = "/tmp/rsa-server-pub-key.pem";
	private static final String HALLO_WELT = "Hallo Welt";

	/**
	 * Can be used to generate an RSA keypair.
	 * 
	 * @throws Exception
	 */
	@Ignore("As of being a workshop no units are tested. Caution: When enabling, files are written to the FS!")
	@Test
	public void testAsysmetricCryptography() throws Exception {

		// WARMUP:
		// RSA theRsa = new RSA();
		// KeyPair theServerKeyPair = theRsa.generateKeys( 2048 );
		// KeyPair theServiceKeyPair = theRsa.generateKeys( 2048 );
		KeyPair theServerKeyPair = PublicKeyUtils.generate( "RSA", 2048 );
		KeyPair theServiceKeyPair = PublicKeyUtils.generate( "RSA", 2048 );

		PublicKey theServerPublicKey = theServerKeyPair.getPublic();
		PrivateKey theServerPrivateKey = theServerKeyPair.getPrivate();

		// @formatter:off
		 CryptWriter.writePemKey( theServerKeyPair.getPrivate(), null, new SecureRandom(), new File( PATH_TO_SERVER_KEY ) );
		 CryptWriter.writePemKey( theServerKeyPair.getPublic(), new File( PATH_TO_SERVER_PUB_KEY ) );
		 CryptWriter.writePemKey( theServiceKeyPair.getPrivate(), null, new SecureRandom(), new File( PATH_TO_SERVICE_KEY ) );
		 CryptWriter.writePemKey( theServiceKeyPair.getPublic(), new File( PATH_TO_SERVICE_PUB_KEY ) );
		// @formatter:on

		AsymmetricAlgorithm theEncryptAlgorithm = new RSA();
		theEncryptAlgorithm.setKey( theServerPublicKey );
		theEncryptAlgorithm.initEncrypt();

		AsymmetricAlgorithm theDencryptAlgorithm = new RSA();
		theDencryptAlgorithm.setKey( theServerPrivateKey );
		theDencryptAlgorithm.initDecrypt();

		Base64Converter theBase64Converter = new Base64Converter();
		String theEncryptedText = theEncryptAlgorithm.encrypt( HALLO_WELT.getBytes(), theBase64Converter );
		LOGGER.info( "Encrypted  = " + theEncryptedText );
		LOGGER.info( "Length     = " + theEncryptedText.length() );

		String theDecryptedText = new String( theDencryptAlgorithm.decrypt( theEncryptedText, theBase64Converter ) );
		LOGGER.info( "Dencrypted = " + theDecryptedText );

		// TESTSTRING:
		String theTestString = "0123456789 0123456789 0123456789 0123456789 0123456789 0123456789 0123456789";

		// ENCRYPTION:
		long theStartTime = System.currentTimeMillis();
		int theRuns = 1000;
		for ( int i = 0; i < theRuns; i++ ) {
			theEncryptedText = theEncryptAlgorithm.encrypt( (theTestString + i).getBytes(), theBase64Converter );
		}
		long theEndTime = System.currentTimeMillis();
		LOGGER.info( "ASYMETRIC ENCRYPTION:" );
		LOGGER.info( theRuns + " / seconds = " + ((theEndTime - theStartTime) / 1000) );
		LOGGER.info( "1 / milliseconds = " + (((double) theEndTime - (double) theStartTime) / theRuns) );

		// DECRYPTION:
		theStartTime = System.currentTimeMillis();
		for ( int i = 0; i < theRuns; i++ ) {
			theDecryptedText = new String( theDencryptAlgorithm.decrypt( theEncryptedText, theBase64Converter ) );
		}
		theEndTime = System.currentTimeMillis();
		LOGGER.info( "ASYMETRIC DECRYPTION:" );
		LOGGER.info( theRuns + " / seconds = " + ((theEndTime - theStartTime) / 1000) );
		LOGGER.info( "1 / milliseconds = " + (((double) theEndTime - (double) theStartTime) / theRuns) );
	}
}
