// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.generator.generators.impls;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Random;

import org.refcodes.data.consts.TextConsts;
import org.refcodes.generator.generators.Generator;
import org.refcodes.generator.generators.IdGenerator;
import org.refcodes.numerical.utils.NumericalUtility;

/**
 * The {@link UniqueIdGeneratorImpl} is an implementation of the
 * {@link Generator} capable of generating ID {@link String} instances unique on
 * the system on which them were generated to IDs generated with the same
 * {@link UniqueIdGeneratorImpl} on another system.
 */
public class UniqueIdGeneratorImpl implements IdGenerator {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Default length for a UUID, Java's UUID class returns strings with a
	 * length of 36.
	 */
	private static final int UUID_LENGTH = 36;

	// /////////////////////////////////////////////////////////////////////////
	// ENUM:
	// /////////////////////////////////////////////////////////////////////////

	public enum Base {
		/**
		 * Alphanumeric with mixed case:
		 */
		BASE_64,
		/**
		 * Alphanumeric in lower case:
		 */
		BASE_36
	}

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Base _base;
	private int _idLength;

	// /////////////////////////////////////////////////////////////////////////
	// STATIC:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * A server in the Internet has its unique IP so that we do not get ID
	 * conflicts for different servers.
	 */
	private static byte[] IP_ADDRESS = new byte[2];
	private static long COUNTER = 0;
	private static long SEED;

	static {
		initSeed();
	}

	/**
	 * This method initializes the ID generator, it is invoked upon loading the
	 * utility class by the class loader (static block), though it may be called
	 * manually in case of ID collisions. Then a new random number as starting
	 * point for a specific part of the ID and the IP address are newly set.
	 */
	public static void initSeed() {
		SEED = new Random().nextLong();
		try {
			byte[] ipAddress = InetAddress.getLocalHost().getAddress();
			IP_ADDRESS[0] = ipAddress[ipAddress.length - 2];
			IP_ADDRESS[1] = ipAddress[ipAddress.length - 1];
		}
		catch ( UnknownHostException e ) {
			new Random().nextBytes( IP_ADDRESS );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link UniqueIdGeneratorImpl} with a default number
	 * {@link Base#BASE_64} to be used for the ID and the given ID length of 36.
	 */
	public UniqueIdGeneratorImpl() {
		this( Base.BASE_64, UUID_LENGTH );
	}

	/**
	 * Constructs a {@link UniqueIdGeneratorImpl} with a default number
	 * {@link Base#BASE_64} to be used for the ID and the provided ID length.
	 * 
	 * @param aIdLength The length for the generated IDs.
	 */
	public UniqueIdGeneratorImpl( int aIdLength ) {
		this( Base.BASE_64, aIdLength );
	}

	/**
	 * Constructs a {@link UniqueIdGeneratorImpl} with the given number
	 * {@link Base} to be used for the ID and the given ID length.
	 * 
	 * @param aBase The number {@link Base} to be used for the generated IDs.
	 * @param aIdLength The length for the generated IDs.
	 */
	public UniqueIdGeneratorImpl( Base aBase, int aIdLength ) {
		_base = aBase;
		_idLength = aIdLength;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasNext() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String next() {
		String theIpPrefix = null;
		if ( _base.equals( Base.BASE_36 ) ) {
			theIpPrefix = NumericalUtility.toBase36( IP_ADDRESS );
			theIpPrefix = theIpPrefix.toLowerCase();
		}
		else {
			theIpPrefix = NumericalUtility.toBase64( IP_ADDRESS );
		}

		String thePrefix = theIpPrefix.replaceAll( "=", "" );

		if ( _idLength <= thePrefix.length() ) { throw new IllegalArgumentException( "The deisred length of <" + _idLength + "> is too short as the prefix \"" + thePrefix + "\" has already a length of <" + thePrefix.length() ); }

		Long theTime = new Long( System.currentTimeMillis() );
		String theTimeString = null;

		if ( _base.equals( Base.BASE_36 ) ) {
			theTimeString = new String( NumericalUtility.toBase36( theTime ) );
			theTimeString = theTimeString.toLowerCase();
		}
		else {

			theTimeString = NumericalUtility.toBase64( theTime );
		}

		theTimeString = theTimeString.replaceAll( "=", "" );

		String theSeedString = null;

		while ( theSeedString == null || theSeedString.length() == 0 ) {
			if ( _base.equals( Base.BASE_36 ) ) {
				theSeedString = new String( NumericalUtility.toBase36( SEED++ ) );
				theSeedString = theSeedString.toLowerCase();
			}
			else {
				theSeedString = new String( NumericalUtility.toBase64( SEED++ ) );
			}
			theSeedString = theSeedString.replaceAll( "=", "" );
			if ( theSeedString == null || theSeedString.length() == 0 ) {
				initSeed();
			}
		}

		if ( theSeedString.length() > 4 ) {
			theSeedString = theSeedString.substring( theSeedString.length() - 4 );
		}
		int theRemainingLength = (_idLength - theSeedString.length()) - thePrefix.length();

		// -----------------------------------
		// Everything fits without filling up:
		// -----------------------------------
		if ( theTimeString.length() >= theRemainingLength ) { return thePrefix + theSeedString + theTimeString.substring( theTimeString.length() - theRemainingLength ); }

		// ------------------------------
		// We have to fill the string up:
		// ------------------------------
		String theCounterString;
		if ( _base.equals( Base.BASE_36 ) ) {
			theCounterString = new String( NumericalUtility.toBase36( COUNTER++ ) );
		}
		else {
			theCounterString = new String( NumericalUtility.toBase64( COUNTER++ ) );
		}
		theCounterString = theCounterString.replaceAll( "=", "" );

		// "theCounterString = AlignTextUtility.toAlignLeft( theCounterString, theRemainingLength, '0' );"
		// AlignTextUtility.toAlignLeft |-->
		if ( theCounterString.length() != theRemainingLength ) {
			if ( theCounterString.length() > theRemainingLength ) {
				theCounterString = theCounterString.substring( 0, theRemainingLength );
			}
			else {
				StringBuffer theBuffer = new StringBuffer();
				theBuffer.append( theCounterString );
				for ( int i = 0; i < theRemainingLength - theCounterString.length(); i++ ) {
					theBuffer.append( '0' );
				}
				theCounterString = theBuffer.toString();
			}
		}
		// <--| AlignTextUtility.toAlignLeft

		return thePrefix + theSeedString + theTimeString + theCounterString;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove() {
		throw new UnsupportedOperationException( TextConsts.UNSUPPORTED_OPERATION );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Don't want a "refcodes-textual" dependency. Therefore the code is hacked
	 * (duplicated) here:
	 * 
	 * @see AlignTextUtility.toAlignLeft
	 */
	// @formatter:off
	// private static String toAlignLeft( String aText, int aLength, char aFillChar ) {
	//	if ( aText.length() == aLength ) { return aText; }
	//	if ( aText.length() > aLength ) { return aText.substring( 0, aLength ); }
	//	StringBuffer theBuffer = new StringBuffer();
	//	theBuffer.append( aText );
	//	for ( int i = 0; i < aLength - aText.length(); i++ ) {
	//		theBuffer.append( aFillChar );
	//	}
	//	return theBuffer.toString();
	// }
	// @formatter:on
}
