// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.IOException;
import java.io.InputStream;

import org.refcodes.io.InputStreamAccessor.InputStreamBuilder;
import org.refcodes.io.InputStreamAccessor.InputStreamProperty;
import org.refcodes.mixin.EncodingAccessor.EncodingBuilder;
import org.refcodes.mixin.EncodingAccessor.EncodingProperty;

/**
 * The {@link InputStreamStringBuilder} constructs {@link String} instances from
 * {@link InputStream} instances. Either use
 * {@link #setInputStream(InputStream)} ({@link #withInputStream(InputStream)})
 * followed by a {@link Object#toString()} method (not thread safe) or directly
 * call {@link #toString(InputStream)} (thread safe). You may specify an
 * encoding to be used such as UTF-8 by either setting the encoding attribute
 * with {@link #setEncoding(String)} ({@link #withEncoding(String)}) or by
 * passing the encoding to the conversion method as of {@link #toString(String)}
 * which overrules the encoding attribute.
 */
public interface InputStreamStringBuilder extends InputStreamProperty, InputStreamBuilder<InputStreamStringBuilder>, EncodingProperty, EncodingBuilder<InputStreamStringBuilder> {

	/**
	 * {@inheritDoc}
	 */
	@Override
	default InputStreamStringBuilder withEncoding( String aEncoding ) {
		setEncoding( aEncoding );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default InputStreamStringBuilder withInputStream( InputStream aInputStream ) {
		setInputStream( aInputStream );
		return this;
	}

	/**
	 * The {@link String} being build by the builder upon the settings of the
	 * attributes.
	 * 
	 * @param aInputStream The {@link InputStream} which to convert into a
	 *        {@link String}.
	 * 
	 * @return The according resulting {@link String}
	 * 
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	default String toString( InputStream aInputStream ) throws IOException {
		return toString( aInputStream, getEncoding() );
	}

	/**
	 * The {@link String}s being build by the builder upon the settings of the
	 * attributes.
	 * 
	 * @param aInputStream The {@link InputStream} which to convert into a
	 *        {@link String} array.
	 * @return The according resulting {@link String} array
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	default String[] toStrings( InputStream aInputStream ) throws IOException {
		return toString( aInputStream, getEncoding() ).split( "\\r\\n|\\n|\\r" );
	}

	/**
	 * The {@link String}s being build by the builder upon the settings of the
	 * attributes.
	 * 
	 * @return The according resulting {@link String} array
	 * 
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	default String[] toStrings() throws IOException {
		return toStrings( getEncoding() );
	}

	/**
	 * The {@link String} being build by the builder upon the settings of the
	 * attributes.
	 * 
	 * @param aInputStream The {@link InputStream} which to convert into a
	 *        {@link String}.
	 * 
	 * @param aEncoding The text encoding to be used.
	 * 
	 * @return The according resulting {@link String}
	 * 
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	String toString( InputStream aInputStream, String aEncoding ) throws IOException;

	/**
	 * The {@link String}s being build by the builder upon the settings of the
	 * attributes.
	 * 
	 * @param aInputStream The {@link InputStream} which to convert into a
	 *        {@link String} array.
	 * 
	 * @param aEncoding The text encoding to be used.
	 * 
	 * @return The according resulting {@link String} array
	 * 
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	default String[] toStrings( InputStream aInputStream, String aEncoding ) throws IOException {
		return toString( aInputStream, aEncoding ).split( "\\r\\n|\\n|\\r" );
	}

	/**
	 * The {@link String}s being build by the builder upon the settings of the
	 * attributes.
	 *
	 * @param aEncoding The text encoding to be used.
	 * @return The according resulting {@link String} array
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	default String[] toStrings( String aEncoding ) throws IOException {
		return toString( aEncoding ).split( "\\r\\n|\\n|\\r" );
	}

	/**
	 * The {@link String}s being build by the builder upon the settings of the
	 * attributes.
	 *
	 * @param aEncoding The text encoding to be used.
	 * @return The according resulting {@link String} array
	 * @throws IOException thrown in case accessing the {@link InputStream}
	 *         caused faults.
	 */
	String toString( String aEncoding ) throws IOException;

}
