// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.IOException;
import java.io.OutputStream;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;

/**
 * Implementation of the {@link OutputStreamConnectionSender} interface.
 */
public class OutputStreamConnectionByteSenderImpl extends AbstractByteSender implements OutputStreamConnectionByteSender {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private OutputStream _outputStream = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void open( OutputStream aOutputStream ) throws OpenException {
		if ( isOpened() ) {
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is " + getConnectionStatus() + "." );
		}
		try {
			_outputStream = aOutputStream;
			_outputStream.flush();
		}
		catch ( IOException aException ) {
			throw new OpenException( "Unable to open the I/O stream receiver as of a causing exception.", aException );
		}
		setConnectionStatus( ConnectionStatus.OPENED );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpenable( OutputStream aOutputStream ) {
		if ( aOutputStream == null ) {
			return false;
		}
		return !isOpened();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		try {
			_outputStream.flush();
		}
		catch ( IOException e ) {
			throw new OpenException( "Unable to flush underlying output stream <" + _outputStream + ">.", e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close() throws CloseException {
		super.close();
		try {
			_outputStream.close();
		}
		catch ( IOException aException ) {}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagram( byte aDatagram ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagram + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		synchronized ( _outputStream ) {
			try {
				_outputStream.write( aDatagram );
				_outputStream.flush();
			}
			catch ( IOException aException ) {
				if ( isClosed() ) {
					return;
				}
				else {
					try {
						close();
					}
					catch ( CloseException e ) {
						throw new OpenException( "Unable to close malfunctioning connection.", e );
					}
				}
				throw new OpenException( aException );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

}
