// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;

import org.refcodes.component.Closable;
import org.refcodes.component.CloseException;
import org.refcodes.component.Flushable;
import org.refcodes.component.OpenException;

/**
 * The {@link SenderDecorator} decorates a {@link Consumer} with the additional
 * methods of a {@link Sender} making it easy to use a {@link Consumer} wherever
 * a {@link Sender} is expected.
 *
 * @param <DATA> the generic type
 */
public class SenderDecorator<DATA extends Serializable> extends AbstractSender<DATA> implements Sender<DATA> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Consumer<DATA> _consumer;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new sender decorator.
	 *
	 * @param aConsumer the consumer
	 */
	public SenderDecorator( Consumer<DATA> aConsumer ) {
		_consumer = aConsumer;
		try {
			open();
		}
		catch ( OpenException ignore ) {}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagram( DATA aDatagram ) throws OpenException {
		_consumer.writeDatagram( aDatagram );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( DATA[] aDatagrams ) throws OpenException {
		_consumer.writeDatagrams( aDatagrams );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( DATA[] aDatagrams, int aOffset, int aLength ) throws OpenException {
		_consumer.writeDatagrams( aDatagrams, aOffset, aLength );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		if ( _consumer instanceof Flushable ) {
			((Flushable) _consumer).flush();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		if ( _consumer instanceof Closable ) {
			((Closable) _consumer).close();
		}
		super.close();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
