// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

/**
 * Ready to use {@link GenericActionEvent} storing a defined set of
 * {@link EventMetaData}.
 * <p>
 * TIPP: In order to distinguish {@link AbstractActionMetaDataEvent} instances
 * from each other, create an actions enumeration, enumerating the various event
 * actions you support. Pass the actual action you intend to notify upon to the
 * according constructor, as an {@link Observer} you may use the declarative
 * method {@link EventMatcherSugar#actionEqualWith(Object)} to test whether your
 * action was notified (or a simple switch case statement).
 * 
 * @param <A> The type of the action stored in the event.
 * @param <EM> The type of the EventMetaData
 * @param <SRC> The type of the source in question.
 */
public abstract class AbstractActionMetaDataEvent<A, EM extends EventMetaData, SRC> extends AbstractActionEvent<A, SRC> implements GenericActionMetaDataEvent<A, EM, SRC> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected EM _eventMetaData;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractActionMetaDataEvent( EM aEventMetaData, SRC aSource ) {
		super( aSource );
		_eventMetaData = aEventMetaData;
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aAction The action which the event represents.
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractActionMetaDataEvent( A aAction, EM aEventMetaData, SRC aSource ) {
		super( aAction, aSource );
		_eventMetaData = aEventMetaData;
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 *
	 * @param aAction The action which the event represents.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractActionMetaDataEvent( A aAction, SRC aSource ) {
		super( aAction, aSource );
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 *
	 * @param aSource The source from which this event originated.
	 */
	public AbstractActionMetaDataEvent( SRC aSource ) {
		super( aSource );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EM getMetaData() {
		return _eventMetaData;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return getClass().getSimpleName() + " [eventMetaData=" + _eventMetaData + ", action=" + getAction() + "]";
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Implementation of the {@link GenericActionMetaDataEventBuilder} interface
	 * for easily creating {@link ActionMetaDataEvent} instances.
	 * 
	 * @param <A> The type of the action stored in the event.
	 * @param <EM> The type of the EventMetaData
	 * @param <SRC> The type of the source in question.
	 * @param <B> The type of the builder to be returned by the builder methods.
	 */
	public static abstract class AbstractActionMetaDataEventBuilder<A, EM extends EventMetaData, SRC, B extends GenericActionMetaDataEventBuilder<A, EM, SRC, B>> extends AbstractActionMetaDataEvent<A, EM, SRC> implements GenericActionMetaDataEventBuilder<A, EM, SRC, B> {

		// /////////////////////////////////////////////////////////////////////////
		// STATICS:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// CONSTANTS:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aEventMetaData The Meta-Data to by supplied by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionMetaDataEventBuilder( EM aEventMetaData, SRC aSource ) {
			super( aEventMetaData, aSource );
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aAction The action which the event represents.
		 * @param aEventMetaData The Meta-Data to by supplied by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionMetaDataEventBuilder( A aAction, EM aEventMetaData, SRC aSource ) {
			super( aAction, aEventMetaData, aSource );
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 *
		 * @param aAction The action which the event represents.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionMetaDataEventBuilder( A aAction, SRC aSource ) {
			super( aAction, aSource );
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 *
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionMetaDataEventBuilder( SRC aSource ) {
			super( aSource );
		}

		// /////////////////////////////////////////////////////////////////////////
		// INJECTION:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Class<?> getPublisherType() {
			if ( _eventMetaData == null ) { return null; }
			return ((EventMetaDataBuilder) _eventMetaData).getPublisherType();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getUniversalId() {
			if ( _eventMetaData == null ) { return null; }
			return ((EventMetaDataBuilder) _eventMetaData).getUniversalId();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getAlias() {
			if ( _eventMetaData == null ) { return null; }
			return ((EventMetaDataBuilder) _eventMetaData).getAlias();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getGroup() {
			if ( _eventMetaData == null ) { return null; }
			return ((EventMetaDataBuilder) _eventMetaData).getGroup();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getChannel() {
			if ( _eventMetaData == null ) { return null; }
			return ((EventMetaDataBuilder) _eventMetaData).getChannel();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void setAction( A aAction ) {
			_action = aAction;
		}

		// /////////////////////////////////////////////////////////////////////////
		// HOOKS:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// HELPER:
		// /////////////////////////////////////////////////////////////////////////

		// /////////////////////////////////////////////////////////////////////////
		// INNER CLASSES:
		// /////////////////////////////////////////////////////////////////////////

	}

}
