// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.ext.eureka;

import java.util.concurrent.ExecutorService;

import org.refcodes.component.CloseException;
import org.refcodes.component.InitializeException;
import org.refcodes.component.LifeCycleStatus;
import org.refcodes.component.PauseException;
import org.refcodes.component.ResumeException;
import org.refcodes.component.StartException;
import org.refcodes.component.StopException;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.net.LoadBalancingStrategy;
import org.refcodes.net.Url;
import org.refcodes.rest.AbstractHttpDiscoveryRestClientDecorator;
import org.refcodes.rest.HttpRestClient;
import org.refcodes.rest.ext.eureka.EurekaDiscoverySidecarImpl.RefreshDaemon;
import org.refcodes.security.TrustStoreDescriptor;

/**
 * The {@link EurekaRestClientDecorator} decorates a {@link HttpRestClient} with
 * functionality such registering and unregistering from / to a Eureka discovery
 * service.
 */
public class EurekaRestClientDecorator extends AbstractHttpDiscoveryRestClientDecorator<EurekaRestClient> implements EurekaRestClient {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private ExecutorService _executorService;
	private RefreshDaemon _refreshDaemon;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Decorates the given {@link HttpRestClient} with discovery functionality.
	 * 
	 * @param aClient The {@link HttpRestClient} to be decorated.
	 */
	public EurekaRestClientDecorator( HttpRestClient aClient ) {
		super( aClient );
	}

	/**
	 * Decorates the given {@link HttpRestClient} with discovery functionality.
	 * 
	 * @param aClient The {@link HttpRestClient} to be decorated.
	 * 
	 * @param aExecutorService An executor service to be used when creating
	 *        {@link Thread}s.
	 */
	public EurekaRestClientDecorator( HttpRestClient aClient, ExecutorService aExecutorService ) {
		super( aClient );
		_executorService = aExecutorService;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// LIFE-CYCLE:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void initialize( Url aDiscoveryUrl, LoadBalancingStrategy aStrategy, TrustStoreDescriptor aStoreDescriptor ) throws InitializeException {
		aDiscoveryUrl = toHttpDiscoveryUrl( aDiscoveryUrl );
		aStrategy = toLoadBalancerStrategy( aStrategy );
		super.initialize();
		setLoadBalancingStrategy( aStrategy );
		try {
			_refreshDaemon = new RefreshDaemon( aDiscoveryUrl, aStoreDescriptor, this, _executorService );
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new InitializeException( ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public synchronized void start() throws StartException {
		try {
			if ( !isConnectionOpened() ) {
				LOGGER.warn( "This connection not opened yet (it is in status <" + getConnectionStatus() + ">, therefore will try to open this connection now..." );
				open();
			}
			super.start();
			_refreshDaemon.start();
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new StartException( ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public synchronized void pause() throws PauseException {
		super.pause();
		try {}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new PauseException( ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public synchronized void stop() throws StopException {
		super.stop();
		try {
			_refreshDaemon.start();
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new StopException( ExceptionUtility.toMessage( e ), e );
		}

	}

	@Override
	public synchronized void resume() throws ResumeException {
		super.resume();
		try {}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new ResumeException( ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public synchronized void destroy() {
		super.destroy();
		try {
			_refreshDaemon.destroy();
		}
		catch ( Exception e ) {
			LOGGER.warn( ExceptionUtility.toMessage( e ), e );
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );

		}
		finally {
			try {
				close();
			}
			catch ( CloseException e ) {
				LOGGER.warn( ExceptionUtility.toMessage( e ), e );
				_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Url toUrl( Url aUrl ) {
		return EurekaDiscoverySidecarImpl.toUrl( aUrl, this, _refreshDaemon );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
