// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.ext.eureka;

import java.net.UnknownHostException;
import java.util.Timer;
import java.util.concurrent.ExecutorService;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.InitializeException;
import org.refcodes.component.LifeCycleStatus;
import org.refcodes.component.OpenException;
import org.refcodes.component.PauseException;
import org.refcodes.component.ResumeException;
import org.refcodes.component.StartException;
import org.refcodes.component.StopException;
import org.refcodes.data.Scheme;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.MediaType;
import org.refcodes.net.Url;
import org.refcodes.net.UrlImpl;
import org.refcodes.rest.AbstractHttpRegistryRestServerDecorator;
import org.refcodes.rest.HttpExceptionHandler;
import org.refcodes.rest.HttpExceptionHandling;
import org.refcodes.rest.HttpRestServer;
import org.refcodes.rest.RestRequestObserver;
import org.refcodes.security.TrustStoreDescriptor;

/**
 * The {@link EurekaRestServerDecorator} implements the {@link EurekaRestServer}
 * interface and decorates a given {@link HttpRestServer} with functionality
 * such registering and unregistering from or to an Eureka discovery service.
 * 
 * Follow the documentation of the {@link EurekaRestServer} in order to initiate
 * the states such as {@link EurekaServiceStatus#UP},
 * {@link EurekaServiceStatus#DOWN} or
 * {@link EurekaServiceStatus#OUT_OF_SERVICE} and unregistering.
 */
public class EurekaRestServerDecorator extends AbstractHttpRegistryRestServerDecorator<EurekaServerDescriptor, EurekaRestServer> implements EurekaRestServer {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private ExecutorService _executorService;
	private Timer _scheduler;
	private EurekaDataCenterType _dataCenterType;
	private String _statusPath;
	private String _homePath;
	private RestRequestObserver _homeRequestObserver;
	private RestRequestObserver _statusRequestObserver;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Decorates the given {@link HttpRestServer} with discovery functionality.
	 * Use {@link #initialize()}, {@link #start()}, {@link #pause()},
	 * {@link #resume()}, {@link #stop()} and {@link #destroy()} for publishing
	 * status updates to Eureka. Use
	 * {@link #open(org.refcodes.net.HttpServerContext)} or similar to activate
	 * your server.
	 * 
	 * @param aServer The {@link HttpRestServer} to be decorated.
	 */
	public EurekaRestServerDecorator( HttpRestServer aServer ) {
		super( aServer );
	}

	/**
	 * Decorates the given {@link HttpRestServer} with discovery functionality.
	 * Use {@link #initialize()}, {@link #start()}, {@link #pause()},
	 * {@link #resume()}, {@link #stop()} and {@link #destroy()} for publishing
	 * status updates to Eureka. Use
	 * {@link #open(org.refcodes.net.HttpServerContext)} or similar to activate
	 * your server.
	 * 
	 * @param aServer The {@link HttpRestServer} to be decorated.
	 * 
	 * @param aExecutorService An executor service to be used when creating
	 *        {@link Thread}s.
	 */
	public EurekaRestServerDecorator( HttpRestServer aServer, ExecutorService aExecutorService ) {
		super( aServer );
		_executorService = aExecutorService;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void initialize( EurekaServerDescriptor aServerDescriptor, RestRequestObserver aPingObserver, RestRequestObserver aStatusObserver, RestRequestObserver aHomeObserver, Url aRegistryUrl, TrustStoreDescriptor aStoreDescriptor ) throws InitializeException {
		super.initialize();

		aPingObserver = toPingObserver( aPingObserver );
		aHomeObserver = toHomeObserver( aHomeObserver );
		aStatusObserver = toStatusObserver( aStatusObserver );
		aRegistryUrl = toHttpRegistryUrl( aRegistryUrl );
		aStoreDescriptor = toTrustStoreDescriptor( aStoreDescriptor );
		aServerDescriptor = toHttpServerDescriptor( aServerDescriptor );

		try {
			if ( aPingObserver != null ) {
				if ( aServerDescriptor.getPingUrl() == null ) { throw new InitializeException( "Cannot register a PING observer without a PING path being defined in the server descriptor!" ); }
				onGet( aServerDescriptor.getPingUrl().getPath(), aPingObserver ).open();
			}
			if ( aStatusObserver != null ) {
				if ( aServerDescriptor.getStatusUrl() == null ) { throw new InitializeException( "Cannot register a STATUS observer without a STATUS path being defined in the server descriptor!" ); }
				onGet( aServerDescriptor.getStatusUrl().getPath(), aStatusObserver ).open();
			}
			if ( aHomeObserver != null ) {
				if ( aServerDescriptor.getHomeUrl() == null ) { throw new InitializeException( "Cannot register a HOME observer without a HOME path being defined in the server descriptor!" ); }
				onGet( aServerDescriptor.getHomeUrl().getPath(), aHomeObserver ).open();
			}
			disableObservers();
			doRegister( EurekaServiceStatus.STARTING );
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new InitializeException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void start() throws StartException {
		try {
			if ( !isConnectionOpened() ) {
				LOGGER.warn( "This connection not opened yet (it is in status <" + getConnectionStatus() + ">, therefore will try to open this connection now..." );
				open();
			}
			super.start();
			enableObservers();
			doStatusUpdate( EurekaServiceStatus.UP );
			_scheduler = new Timer( true );
			_scheduler.schedule( new EurekaRegistrySidecarImpl.HeartBeatDaemon( this, _executorService ), EurekaLoopSleepTime.REGISTRY_SERVICE_HEARBEAT.getMilliseconds(), EurekaLoopSleepTime.REGISTRY_SERVICE_HEARBEAT.getMilliseconds() );
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new StartException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void pause() throws PauseException {
		super.pause();
		try {
			doStatusUpdate( EurekaServiceStatus.DOWN );
			disableObservers();
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new PauseException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void resume() throws ResumeException {
		super.resume();
		try {
			enableObservers();
			doStatusUpdate( EurekaServiceStatus.UP );
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new ResumeException( ExceptionUtility.toMessage( e ), e );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void stop() throws StopException {
		super.stop();
		try {
			_scheduler.cancel();
			doStatusUpdate( EurekaServiceStatus.OUT_OF_SERVICE );
			disableObservers();
		}
		catch ( Exception e ) {
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			throw new StopException( ExceptionUtility.toMessage( e ), e );
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void destroy() {
		super.destroy();
		try {
			_scheduler.cancel();
			doDeregister();
		}
		catch ( Exception e ) {
			LOGGER.warn( ExceptionUtility.toMessage( e ), e );
			_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );

		}
		finally {
			try {
				disableObservers();
				close();
			}
			catch ( CloseException e ) {
				LOGGER.warn( ExceptionUtility.toMessage( e ), e );
				_lifeCycleAutomaton.setLifeCycleStatus( LifeCycleStatus.ERROR );
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getHomePath() {
		return _homePath;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHomePath( String aHomePath ) {
		_homePath = aHomePath;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getStatusPath() {
		return _statusPath;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setStatusPath( String aStatusPath ) {
		_statusPath = aStatusPath;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _server.getConnectionStatus();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setObserversActive( boolean isActive ) {
		_server.setObserversActive( isActive );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isObserversActive() {
		return _server.isObserversActive();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EurekaRestServer withHttpExceptionHandler( HttpExceptionHandler aHttpErrorHandler ) {
		_server.setHttpExceptionHandler( aHttpErrorHandler );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHttpExceptionHandler( HttpExceptionHandler aHttpErrorHandler ) {
		_server.setHttpExceptionHandler( aHttpErrorHandler );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpExceptionHandler getHttpExceptionHandler() {
		return _server.getHttpExceptionHandler();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EurekaRestServer withHttpExceptionHandling( HttpExceptionHandling aHttpErrorHandling ) {
		_server.setHttpExceptionHandling( aHttpErrorHandling );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHttpExceptionHandling( HttpExceptionHandling aHttpErrorHandling ) {
		_server.setHttpExceptionHandling( aHttpErrorHandling );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpExceptionHandling getHttpExceptionHandling() {
		return _server.getHttpExceptionHandling();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EurekaDataCenterType getEurekaDataCenterType() {
		return _dataCenterType;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEurekaDataCenterType( EurekaDataCenterType aDataCenterType ) {
		_dataCenterType = aDataCenterType;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onHomeRequest( RestRequestObserver aRequestObserver ) {
		_homeRequestObserver = aRequestObserver;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RestRequestObserver getHomeRequestObserver() {
		return _homeRequestObserver;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onStatusRequest( RestRequestObserver aRequestObserver ) {
		_statusRequestObserver = aRequestObserver;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RestRequestObserver getStatusRequestObserver() {
		return _statusRequestObserver;
	}

	// /////////////////////////////////////////////////////////////////////////
	// TWEAKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EurekaServerDescriptor toHttpServerDescriptor( String aAlias, String aInstanceId, Scheme aScheme, String aHost, String aVirtualHost, int[] aIpAddress, int aPort, String aPingPath, String aStatusPath, String aHomePath, EurekaDataCenterType aDataCenterType ) {
		return EurekaRegistrySidecarImpl.toHttpServerDescriptor( aAlias, aInstanceId, aScheme, aHost, aVirtualHost, aIpAddress, aPort, aPingPath, aStatusPath, aHomePath, aDataCenterType, this );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Registers the given service at Eureka.
	 * 
	 * @param aServiceStatus The {@link EurekaServiceStatus} to be set.
	 * @throws HttpStatusException Thrown in case a HTTP response was of an
	 *         erroneous status.t
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	protected void doRegister( EurekaServiceStatus aServiceStatus ) throws HttpStatusException, OpenException {
		EurekaRegistrySidecarImpl.doRegister( aServiceStatus, this, _executorService );
	}

	/**
	 * Does a Eureka status update for the given service.
	 * 
	 * @param aServiceStatus The {@link EurekaServiceStatus} to be set.
	 * @throws HttpStatusException Thrown in case a HTTP response was of an
	 *         erroneous status.t
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 * @throws UnknownHostException Thrown in case the targeted host is unknown.
	 */
	protected void doStatusUpdate( EurekaServiceStatus aServiceStatus ) throws OpenException, UnknownHostException, HttpStatusException {
		EurekaRegistrySidecarImpl.doStatusUpdate( aServiceStatus, this, _executorService );
	}

	/**
	 * Unregisters the given service at Eureka.
	 * 
	 * @throws HttpStatusException Thrown in case a HTTP response was of an
	 *         erroneous status.t
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	protected void doDeregister() throws OpenException, HttpStatusException {
		EurekaRegistrySidecarImpl.doDeregister( this, _executorService );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Resolves the property from the provided value and this instance and sets
	 * the property in case the provided value is not null.
	 * 
	 * @param aRegistryUrl The value to be used when not null.
	 * 
	 * @return The value when not null, else the value of the provided property.
	 */
	@Override
	protected Url toHttpRegistryUrl( Url aRegistryUrl ) {
		aRegistryUrl = super.toHttpRegistryUrl( aRegistryUrl );
		if ( aRegistryUrl != null && aRegistryUrl.getPath() == null ) {
			aRegistryUrl = new UrlImpl( aRegistryUrl, EUREKA_BASE_PATH );
			setHttpRegistryUrl( aRegistryUrl );
		}
		return aRegistryUrl;
	}

	/**
	 * Resolves the property from the provided value and this instance and sets
	 * the property in case the provided value is not null.
	 * 
	 * @param aHomeRequestObserver The value to be used when not null.
	 * 
	 * @return The value when not null, else the value of the provided property.
	 */
	protected RestRequestObserver toHomeObserver( RestRequestObserver aHomeRequestObserver ) {
		if ( aHomeRequestObserver == null ) {
			aHomeRequestObserver = _homeRequestObserver;
			if ( aHomeRequestObserver == null ) {
				aHomeRequestObserver = ( aReq, aResp ) -> {
					aResp.getHeaderFields().putContentType( MediaType.TEXT_PLAIN );
					aResp.setResponse( "Pong!" );
					LOGGER.info( "Received a HOME request, no HOME handler defined, using default handler!" );
				};
				_homeRequestObserver = aHomeRequestObserver;
			}
		}
		return aHomeRequestObserver;
	}

	/**
	 * Resolves the property from the provided value and this instance and sets
	 * the property in case the provided value is not null.
	 * 
	 * @param aStatusRequestObserver The value to be used when not null.
	 * 
	 * @return The value when not null, else the value of the provided property.
	 */
	protected RestRequestObserver toStatusObserver( RestRequestObserver aStatusRequestObserver ) {
		if ( aStatusRequestObserver == null ) {
			aStatusRequestObserver = _statusRequestObserver;
			if ( aStatusRequestObserver == null ) {
				aStatusRequestObserver = ( aReq, aResp ) -> {
					aResp.getHeaderFields().putContentType( MediaType.TEXT_PLAIN );
					aResp.setResponse( "Pong!" );
					LOGGER.info( "Received a STATUS request, no STATUS handler defined, using default handler!" );
				};
				_statusRequestObserver = aStatusRequestObserver;
			}
		}
		return aStatusRequestObserver;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
