package org.refcodes.rest;

import java.io.File;
import java.net.URL;
import java.security.KeyStore;

import org.refcodes.component.LinkComponent;
import org.refcodes.component.OpenException;
import org.refcodes.net.BaseUrlAccessor.BaseUrlBuilder;
import org.refcodes.net.BaseUrlAccessor.BaseUrlProperty;

/**
 * Extends a {@link RestClient} to be capable of providing a User-Agent with
 * {@link #setUserAgent(String)} ({@link #withUserAgent(String)}) and to be
 * capable of using base URLs to be set with {@link #setBaseUrl(String)} (
 * {@link #withBaseUrl(String)}).
 * 
 * To prepare HTTPS connections, use the methods such as:
 * 
 * {@link #open(File, String)} or {@link #open(File, String, String)}.
 * 
 * A {@link HttpRestClient} can be shutdown via {@link #close()}.
 */
public interface HttpRestClient extends RestClient, LinkComponent, BaseUrlProperty, BaseUrlBuilder<HttpRestClient> {

	String DEFAULT_SSL_PROTOCOL = "TLS";

	String DEFAULT_KEYSTORE_FORMAT = "JKS";

	/**
	 * With session correlation.
	 *
	 * @param hasSessionCorrelation the has session correlation
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withSessionCorrelation( boolean hasSessionCorrelation ) {
		setSessionCorrelation( hasSessionCorrelation );
		return this;
	}

	/**
	 * With enable session correlation.
	 *
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withEnableSessionCorrelation() {
		enableSessionCorrelation();
		return this;
	}

	/**
	 * With disable session correlation.
	 *
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withDisableSessionCorrelation() {
		disableSessionCorrelation();
		return this;
	}

	/**
	 * With request correlation.
	 *
	 * @param hasRequestCorrelation the has request correlation
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withRequestCorrelation( boolean hasRequestCorrelation ) {
		setRequestCorrelation( hasRequestCorrelation );
		return this;
	}

	/**
	 * With enable request correlation.
	 *
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withEnableRequestCorrelation() {
		enableRequestCorrelation();
		return this;
	}

	/**
	 * With disable request correlation.
	 *
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withDisableRequestCorrelation() {
		disableRequestCorrelation();
		return this;
	}

	/**
	 * Configures the HTTPS client connection with the provided HTTPS
	 * configuration parameters.
	 * 
	 * @param aTrustStoreFile The file pointing to your {@link KeyStore}.
	 * @param aTrustStoreType Type the type of {@link KeyStore}, e.g. "JKS"
	 *        format.
	 * @param aTrustStorePassword The password for the {@link KeyStore}.
	 * 
	 * @throws OpenException thrown in case something went wrong.
	 */
	void open( File aTrustStoreFile, String aTrustStoreType, String aTrustStorePassword ) throws OpenException;

	/**
	 * Same as {@link #open(File, String, String)} but:
	 * 
	 * <ul>
	 * <li>Type the type of {@link KeyStore} is set to "JKS".
	 * </ul>
	 * 
	 * @param aTrustStoreFile The file pointing to your {@link KeyStore}.
	 * @param aTrustStorePassword The password for the {@link KeyStore}.
	 * 
	 * @throws OpenException thrown in case something went wrong.
	 */
	default void open( File aTrustStoreFile, String aTrustStorePassword ) throws OpenException {
		open( aTrustStoreFile, DEFAULT_KEYSTORE_FORMAT, aTrustStorePassword );
	}

	/**
	 * With user agent.
	 *
	 * @param aUserAgent The user agent
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withUserAgent( String aUserAgent ) {
		setUserAgent( aUserAgent );
		return this;
	}

	/**
	 * With base url.
	 *
	 * @param aBaseUrl The base url
	 * @return The {@link HttpRestClient}
	 */
	@Override
	default HttpRestClient withBaseUrl( URL aBaseUrl ) {
		setBaseUrl( aBaseUrl );
		return this;
	}
}
