package org.refcodes.rest;

import java.io.File;
import java.security.KeyStore;

import javax.net.ssl.SSLContext;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionComponent.ConnectionComponentBuilder;
import org.refcodes.component.OpenException;
import org.refcodes.net.BasicAuthObservable;
import org.refcodes.net.HttpsConnectionRequestObservable;

/**
 * Extends a {@link RestServer} to be capable of opening a server socket on the
 * local host with the provided port number via {@link #open(Object)} or with an
 * additional maximum number of connections via {@link #open(Integer, int)}.
 * 
 * To open a HTTPS port, use the methods such as:
 * 
 * {@link #open(File, String, Integer)} or
 * {@link #open(File, String, String, Integer)} or
 * {@link #open(String, File, String, String, String, Integer, int)}.
 * 
 * A {@link HttpRestServer} can be shutdown via {@link #close()}.
 */
public interface HttpRestServer extends RestServer, ConnectionComponentBuilder<Integer, HttpRestServer>, BasicAuthObservable<HttpRestServer>, HttpsConnectionRequestObservable<HttpRestServer> {

	String DEFAULT_SSL_PROTOCOL = "TLS";

	String DEFAULT_KEYSTORE_FORMAT = "JKS";

	/**
	 * With session correlation.
	 *
	 * @param hasSessionCorrelation the has session correlation
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withSessionCorrelation( boolean hasSessionCorrelation ) {
		setSessionCorrelation( hasSessionCorrelation );
		return this;
	}

	/**
	 * With enable session correlation.
	 *
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withEnableSessionCorrelation() {
		enableSessionCorrelation();
		return this;
	}

	/**
	 * With disable session correlation.
	 *
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withDisableSessionCorrelation() {
		disableSessionCorrelation();
		return this;
	}

	/**
	 * With request correlation.
	 *
	 * @param hasRequestCorrelation the has request correlation
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withRequestCorrelation( boolean hasRequestCorrelation ) {
		setRequestCorrelation( hasRequestCorrelation );
		return this;
	}

	/**
	 * With enable request correlation.
	 *
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withEnableRequestCorrelation() {
		enableRequestCorrelation();
		return this;
	}

	/**
	 * With disable request correlation.
	 *
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withDisableRequestCorrelation() {
		disableRequestCorrelation();
		return this;
	}

	/**
	 * Opens the HTTP sever connection (socket) at the given port allowing the
	 * given number of maximum connections at the same time.
	 * 
	 * @param aPort The port on which to listen for requests.
	 * @param aMaxConnections The number of maximum connections at the same time
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	void open( Integer aPort, int aMaxConnections ) throws OpenException;

	/**
	 * Opens the HTTPS sever connection (socket) at the given port allowing the
	 * given number of maximum connections at the same time using the provided
	 * HTTPS configuration parameters.
	 * 
	 * @param aSecureSocketProtocol The protocol to use, e.g. "TLS".
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStoreType The type of {@link KeyStore}, e.g. "JKS" format.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aKeyPassword The password for the {@link KeyStore}'s key.
	 * @param aPort The port on which to listen for requests.
	 * @param aMaxConnections The number of maximum connections at the same time
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	void open( String aSecureSocketProtocol, File aKeyStoreFile, String aKeyStoreType, String aKeyStorePassword, String aKeyPassword, Integer aPort, int aMaxConnections ) throws OpenException;

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The number of maximum connections at the same time is decided upon by
	 * the implementations's default value.
	 * </ul>
	 * 
	 * @param aSecureSocketProtocol The protocol to use, e.g. "TLS".
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStoreType The type of {@link KeyStore}, e.g. "JKS" format.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aKeyPassword The password for the {@link KeyStore}'s key.
	 * @param aPort The port on which to listen for requests.
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( String aSecureSocketProtocol, File aKeyStoreFile, String aKeyStoreType, String aKeyStorePassword, String aKeyPassword, Integer aPort ) throws OpenException {
		open( aSecureSocketProtocol, aKeyStoreFile, aKeyStoreType, aKeyStorePassword, aKeyPassword, aPort, -1 );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * <li>The password for the {@link KeyStore}'s key is assumed to be the same
	 * as the provided {@link KeyStore} password.
	 * </ul>
	 * 
	 * @param aSecureSocketProtocol The protocol to use, e.g. "TLS".
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aPort The port on which to listen for requests.
	 * @param aMaxConnections The number of maximum connections at the same time
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( String aSecureSocketProtocol, File aKeyStoreFile, String aKeyStorePassword, Integer aPort, int aMaxConnections ) throws OpenException {
		open( aSecureSocketProtocol, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aKeyStorePassword, aKeyStorePassword, aPort, aMaxConnections );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * <li>The password for the {@link KeyStore}'s key is assumed to be the same
	 * as the provided {@link KeyStore} password.
	 * <li>The number of maximum connections at the same time is decided upon by
	 * the implementations's default value.
	 * </ul>
	 * 
	 * @param aSecureSocketProtocol The protocol to use, e.g. "TLS".
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aPort The port on which to listen for requests.
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( String aSecureSocketProtocol, File aKeyStoreFile, String aKeyStorePassword, Integer aPort ) throws OpenException {
		open( aSecureSocketProtocol, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aKeyStorePassword, aKeyStorePassword, aPort, -1 );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The protocol fore the underlying {@link SSLContext} used is set to
	 * "TLS".
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * <li>The password for the {@link KeyStore}'s key is assumed to be the same
	 * as the provided {@link KeyStore} password.
	 * </ul>
	 * 
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aPort The port on which to listen for requests.
	 * @param aMaxConnections The number of maximum connections at the same time
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( File aKeyStoreFile, String aKeyStorePassword, Integer aPort, int aMaxConnections ) throws OpenException {
		open( DEFAULT_SSL_PROTOCOL, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aKeyStorePassword, aKeyStorePassword, aPort, aMaxConnections );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The protocol fore the underlying {@link SSLContext} used is set to
	 * "TLS".
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * </ul>
	 * 
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aKeyPassword The password for the {@link KeyStore}'s key.
	 * @param aPort The port on which to listen for requests.
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( File aKeyStoreFile, String aKeyStorePassword, String aKeyPassword, Integer aPort ) throws OpenException {
		open( DEFAULT_SSL_PROTOCOL, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aKeyStorePassword, aKeyPassword, aPort );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The protocol fore the underlying {@link SSLContext} used is set to
	 * "TLS".
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * </ul>
	 * 
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aKeyStorePassword The password for the {@link KeyStore}.
	 * @param aKeyPassword The password for the {@link KeyStore}'s key.
	 * @param aPort The port on which to listen for requests.
	 * @param aMaxConnections The number of maximum connections at the same time
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( File aKeyStoreFile, String aKeyStorePassword, String aKeyPassword, Integer aPort, int aMaxConnections ) throws OpenException {
		open( DEFAULT_SSL_PROTOCOL, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aKeyStorePassword, aKeyPassword, aPort, aMaxConnections );
	}

	/**
	 * Same as {@link #open(String, File, String, String, String, Integer, int)}
	 * but:
	 * 
	 * <ul>
	 * <li>The protocol fore the underlying {@link SSLContext} used is set to
	 * "TLS".
	 * <li>The type of {@link KeyStore} is set to "JKS".
	 * <li>The password for the {@link KeyStore}'s key is assumed to be the same
	 * as the provided {@link KeyStore} password.
	 * <li>The number of maximum connections at the same time is decided upon by
	 * the implementations's default value.
	 * </ul>
	 * 
	 * @param aKeyStoreFile The file pointing to your {@link KeyStore}.
	 * @param aPassword The password for the {@link KeyStore}.
	 * @param aPort The port on which to listen for requests.
	 * 
	 * @throws OpenException thrown in case something went wrong such as the
	 *         port being already in use.
	 */
	default void open( File aKeyStoreFile, String aPassword, Integer aPort ) throws OpenException {
		open( DEFAULT_SSL_PROTOCOL, aKeyStoreFile, DEFAULT_KEYSTORE_FORMAT, aPassword, aPassword, aPort, -1 );
	}

	/**
	 * With realm.
	 *
	 * @param aRealm the a realm
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	/**
	 * With base locator.
	 *
	 * @param aBaseLocator the a base locator
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withBaseLocator( String aBaseLocator ) {
		setBaseLocator( aBaseLocator );
		return this;
	}

	/**
	 * With close.
	 *
	 * @return the http rest server
	 * @throws CloseException the close exception
	 */
	@Override
	default HttpRestServer withClose() throws CloseException {
		close();
		return this;
	}

	/**
	 * With close quietly.
	 *
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withCloseQuietly() {
		closeQuietly();
		return this;
	}

	/**
	 * With close in.
	 *
	 * @param aCloseInMillis the a close in millis
	 * @return the http rest server
	 */
	@Override
	default HttpRestServer withCloseIn( int aCloseInMillis ) {
		closeIn( aCloseInMillis );
		return this;
	}

	/**
	 * With open.
	 *
	 * @param aConnection the a connection
	 * @return the http rest server
	 * @throws OpenException the open exception
	 */
	@Override
	default HttpRestServer withOpen( Integer aConnection ) throws OpenException {
		open( aConnection );
		return this;
	}
}
